/**
 * RMS Theme Accessibility Enhancements
 * WCAG 2.4.1 - Add main landmark to content area
 */
document.addEventListener( 'DOMContentLoaded', function() {

    // Ensure there is a main landmark
    const main = document.querySelector( 'main' );
    if ( main && ! main.hasAttribute( 'id' ) ) {
        main.setAttribute( 'id', 'main' );
    }

    // Normalize Navigation block ARIA and fix list structure
    function normalizeNavARIA( root ) {
        const scope = root || document;

        // Fix invalid list structure: wrap orphaned nested ULs in LI elements
        scope.querySelectorAll( '.wp-block-navigation__container' ).forEach( function( container ) {

            // Find any direct UL children (should be wrapped in LI)
            const directULs = Array.from( container.children ).filter( function( child ) {
                return 'UL' === child.tagName;
            });

            directULs.forEach( function( orphanedUL ) {

                // Create a wrapper LI element
                const wrapperLI = document.createElement( 'li' );
                wrapperLI.className = 'wp-block-navigation-item wp-block-navigation-submenu';

                // Insert the wrapper before the orphaned UL, then move UL inside
                container.insertBefore( wrapperLI, orphanedUL );
                wrapperLI.appendChild( orphanedUL );
            });
        });

        // Target all navigation-related elements with menubar roles (including nested wp-block-page-list)
        scope.querySelectorAll( '.wp-block-navigation__container, .wp-block-page-list, [role="menubar"]' ).forEach( function( ul ) {
            if ( 'menubar' === ul.getAttribute( 'role' ) ) {
                ul.removeAttribute( 'role' );
            }
            if ( ul.hasAttribute( 'aria-label' ) ) {
                ul.removeAttribute( 'aria-label' );
            }
        });

        // Remove menuitem roles from all navigation links and role="none" from list items
        scope.querySelectorAll( '[role="menuitem"], li[role="none"], .wp-block-navigation-item[role="none"]' ).forEach( function( el ) {
            el.removeAttribute( 'role' );
        });

        // Remove any remaining roles from navigation containers to avoid conflicts
        scope.querySelectorAll( '.wp-block-navigation__container' ).forEach( function( container ) {
            if ( container.hasAttribute( 'role' ) ) {
                container.removeAttribute( 'role' );
            }
        });

        // Ensure nav elements have role="navigation"
        scope.querySelectorAll( 'nav' ).forEach( function( nav ) {
            if ( 'navigation' !== nav.getAttribute( 'role' ) ) {
                nav.setAttribute( 'role', 'navigation' );
            }
        });
    }

    // Initial pass
    normalizeNavARIA( document );

    // Observe dynamic changes, e.g., responsive overlay menu content
    const observer = new MutationObserver( function( mutations ) {
        for ( const m of mutations ) {
            if ( m.addedNodes && m.addedNodes.length ) {
                m.addedNodes.forEach( function( node ) {
                    if ( 1 === node.nodeType ) {
                        normalizeNavARIA( node );
                    }
                });
            }

            // Also normalize when attributes change (WordPress core scripts adding roles)
            if ( 'attributes' === m.type && m.target ) {
                normalizeNavARIA( m.target );
            }
        }
    });

    observer.observe( document.body, {
        childList: true,
        subtree: true,
        attributes: true,
        attributeFilter: [ 'role', 'aria-label' ]
    });

    // Additional normalization after a short delay to catch late WordPress script changes
    setTimeout( function() {
        normalizeNavARIA( document );
    }, 100 );
});
