/**
 * RMS Polylang Language Switcher Block
 * WordPress block for displaying a multilingual language switcher dropdown
 */

( function() {
	'use strict';

	const { registerBlockType } = wp.blocks;
	const { useBlockProps, InspectorControls } = wp.blockEditor;
	const { PanelBody, ToggleControl, SelectControl } = wp.components;
	const { __ } = wp.i18n;
	const { createElement: el } = wp.element;

	registerBlockType( 'rms/polylang-language-switcher', {
		apiVersion: 3,
		title: __( 'Language Switcher', 'rms' ),
		description: __( 'A dropdown language switcher powered by Polylang with full accessibility support.', 'rms' ),
		category: 'widgets',
		icon: 'translation',
		keywords: [ __( 'language', 'rms' ), __( 'polylang', 'rms' ), __( 'translation', 'rms' ), __( 'switcher', 'rms' ), __( 'multilingual', 'rms' ) ],
		textdomain: 'rms',
		supports: {
			html: false,
			align: [ 'left', 'center', 'right' ],
			spacing: {
				margin: true,
				padding: true
			},
			color: {
				text: true,
				background: true,
				link: true
			},
			typography: {
				fontSize: true,
				lineHeight: true
			}
		},
		attributes: {
			showFlags: {
				type: 'boolean',
				default: true
			},
			showNames: {
				type: 'boolean',
				default: true
			},
			dropdownStyle: {
				type: 'string',
				default: 'default'
			},
			hideCurrentLanguage: {
				type: 'boolean',
				default: false
			},
			forceHome: {
				type: 'boolean',
				default: false
			},
			displayCurrentFirst: {
				type: 'boolean',
				default: true
			}
		},
		edit: function({ attributes, setAttributes }) {
			const {
				showFlags,
				showNames,
				dropdownStyle,
				hideCurrentLanguage,
				forceHome,
				displayCurrentFirst
			} = attributes;

			const blockProps = useBlockProps({
				className: `wp-block-rms-polylang-language-switcher language-switcher-${dropdownStyle}`
			});

			// Sample language data for preview
			const sampleLanguages = [
				{ name: 'English', slug: 'en', flag: '🇺🇸', current: true },
				{ name: 'Français', slug: 'fr', flag: '🇫🇷', current: false }
			];

			const currentLang = sampleLanguages.find( lang => lang.current ) || sampleLanguages[0];

			// Inspector Controls
			const inspectorControls = el( InspectorControls, {},
				el( PanelBody, {
					title: __( 'Language Switcher Settings', 'rms' ),
					initialOpen: true
				},
					el( SelectControl, {
						label: __( 'Dropdown Style', 'rms' ),
						value: dropdownStyle,
						options: [
							{ label: __( 'Default', 'rms' ), value: 'default' },
							{ label: __( 'Minimal', 'rms' ), value: 'minimal' },
							{ label: __( 'Flags Only', 'rms' ), value: 'flags-only' }
						],
						onChange: function( value ) {
							setAttributes({ dropdownStyle: value });
						}
					}),

					el( ToggleControl, {
						label: __( 'Show Flag Icons', 'rms' ),
						checked: showFlags,
						onChange: function( value ) {
							setAttributes({ showFlags: value });
						}
					}),

					el( ToggleControl, {
						label: __( 'Show Language Names', 'rms' ),
						checked: showNames,
						onChange: function( value ) {
							setAttributes({ showNames: value });
						}
					}),

					el( ToggleControl, {
						label: __( 'Hide Current Language', 'rms' ),
						checked: hideCurrentLanguage,
						help: __( 'Hide the current language from the dropdown list', 'rms' ),
						onChange: function( value ) {
							setAttributes({ hideCurrentLanguage: value });
						}
					}),

					el( ToggleControl, {
						label: __( 'Display Current Language First', 'rms' ),
						checked: displayCurrentFirst,
						help: __( 'Show the current language at the top of the list', 'rms' ),
						onChange: function( value ) {
							setAttributes({ displayCurrentFirst: value });
						}
					}),

					el( ToggleControl, {
						label: __( 'Force Home URL', 'rms' ),
						checked: forceHome,
						help: __( 'Always link to the homepage instead of translated posts', 'rms' ),
						onChange: function( value ) {
							setAttributes({ forceHome: value });
						}
					})
				)
			);

			// Block Content
			const blockContent = el( 'div', blockProps,
				el( 'div', { className: 'language-switcher-preview' },
					el( 'button', {
						type: 'button',
						className: 'language-switcher-button',
						disabled: true,
						style: { cursor: 'default' }
					},
						showFlags && el( 'span', {
							className: 'flag-icon',
							'aria-hidden': 'true'
						}, currentLang.flag ),
						showNames && el( 'span', {
							className: 'language-name'
						}, currentLang.name ),
						el( 'span', {
							className: 'dropdown-arrow',
							'aria-hidden': 'true'
						}, '▼' )
					)

				)
			);

			return [ inspectorControls, blockContent ];
		},

		save: function() {

			// Server-side rendered
			return null;
		}
	});

}() );
