/**
 * RMS Theme JavaScript - Compiled Version
 * JavaScript Development Agent
 *
 * This is the production-ready compiled version with WordPress/jQuery compatibility
 */

( function( $ ) {
    'use strict';

    /**
     * RMS Theme Main Object
     * Uses IIFE pattern to avoid global namespace pollution
     */
    const RMSTheme = {

        /**
         * Initialize theme functionality
         */
        init: function() {
            this.bindEvents();
            this.setupAccessibility();
            this.setupNavigation();
            this.setupLazyLoading();
            this.setupNewsletter();
            this.setupCounters();
            this.adjustMainContentSpacing();
            console.log( 'RMS Theme initialized' );
        },

        /**
         * Bind event listeners
         */
        bindEvents: function() {
            const self = this;

            // Scroll events with throttling for performance
            let scrollTimer = null;
            $( window ).on( 'scroll', function() {
                if ( scrollTimer ) {
                    clearTimeout( scrollTimer );
                }
                scrollTimer = setTimeout( function() {
                    self.handleScroll();
                }, 16 ); // ~60fps
            });

            // Resize events with debouncing
            let resizeTimer = null;
            $( window ).on( 'resize', function() {
                if ( resizeTimer ) {
                    clearTimeout( resizeTimer );
                }
                resizeTimer = setTimeout( function() {
                    self.handleResize();
                }, 250 );
            });

            // Navigation toggle events
            $( document ).on( 'click', '.wp-block-navigation__responsive-container-open', function( e ) {
                e.preventDefault();
                self.openMobileNav();
            });

            $( document ).on( 'click', '.wp-block-navigation__responsive-container-close', function( e ) {
                e.preventDefault();
                self.closeMobileNav();
            });

            // Smooth scroll for anchor links
            $( document ).on( 'click', 'a[href^="#"]', function( e ) {
                const target = $( this.getAttribute( 'href' ) );
                if ( target.length ) {
                    e.preventDefault();
                    $( 'html, body' ).animate({
                        scrollTop: target.offset().top - 100
                    }, 800, function() {
                        target.focus();
                    });
                }
            });
        },

        /**
         * Handle scroll events
         */
        handleScroll: function() {
            const scrollTop = $( window ).scrollTop();
            const $body = $( 'body' );

            // Add scrolled class for styling
            if ( 100 < scrollTop ) {
                $body.addClass( 'scrolled' );
            } else {
                $body.removeClass( 'scrolled' );
            }

            // Update skip link target
            this.updateSkipLinkTarget();

            // Show/hide back to top button
            const $backToTop = $( '.back-to-top' );
            if ( 300 < scrollTop ) {
                $backToTop.addClass( 'visible' );
            } else {
                $backToTop.removeClass( 'visible' );
            }
        },

        /**
         * Handle resize events
         */
    handleResize: function() {

            // Update navigation state on resize
            const isMobile = 768 > window.innerWidth;
            $( 'body' ).toggleClass( 'is-mobile', isMobile );

            // Close mobile menu on desktop
            if ( ! isMobile ) {
                this.closeMobileNav();
            }
        },

        /**
         * Setup accessibility enhancements
         * WCAG 2.1 compliance features
         */
        setupAccessibility: function() {

            // WCAG 2.4.3 - Focus order management
            this.setupFocusManagement();

            // WCAG 2.1.1 - Keyboard navigation
            this.setupKeyboardNavigation();

            // WCAG 4.1.3 - Status messages
            this.setupStatusMessages();

            // WCAG 1.4.12 - Text spacing
            this.setupTextSpacing();
        },

        /**
         * Setup focus management for keyboard navigation
         */
        setupFocusManagement: function() {
            const self = this;

            // Skip link functionality
            $( '.skip-link' ).on( 'click', function( e ) {
                e.preventDefault();
                const target = $( this.getAttribute( 'href' ) );
                if ( target.length ) {
                    target.attr( 'tabindex', '-1' ).focus();

                    // Remove tabindex after blur to restore natural tab order
                    target.one( 'blur', function() {
                        $( this ).removeAttr( 'tabindex' );
                    });
                }
            });

            // Focus visible indicators for better keyboard navigation
            $( 'a, button, input, select, textarea, [tabindex]' ).on( 'focus', function() {
                $( this ).addClass( 'has-focus' );
            }).on( 'blur', function() {
                $( this ).removeClass( 'has-focus' );
            });

            // Trap focus in mobile menu when open
            $( document ).on( 'keydown', function( e ) {
                if ( 9 === e.which && $( '.mobile-menu-open' ).length ) { // Tab key
                    self.trapFocusInMobileMenu( e );
                }
            });
        },

        /**
         * Trap focus within mobile menu
         */
        trapFocusInMobileMenu: function( e ) {
            const $menu = $( '.wp-block-navigation' );
            const $focusableElements = $menu.find( 'a, button, input, select, textarea, [tabindex]:not([tabindex="-1"])' );
            const $firstElement = $focusableElements.first();
            const $lastElement = $focusableElements.last();

            if ( e.shiftKey ) {
                if ( document.activeElement === $firstElement[0]) {
                    e.preventDefault();
                    $lastElement.focus();
                }
            } else {
                if ( document.activeElement === $lastElement[0]) {
                    e.preventDefault();
                    $firstElement.focus();
                }
            }
        },

        /**
         * Setup keyboard navigation
         */
        setupKeyboardNavigation: function() {
            const self = this;

            // ESC key handling for mobile menu
            $( document ).on( 'keydown', function( e ) {
                if ( 27 === e.which ) { // Escape key
                    if ( $( '.mobile-menu-open' ).length ) {
                        self.closeMobileNav();
                    }
                }
            });

            // Arrow key navigation for menus
            $( '.wp-block-navigation a' ).on( 'keydown', function( e ) {
                self.handleMenuKeyNavigation( e );
            });

            // Enter and space for button-like elements
            $( '.wp-block-button__link' ).on( 'keydown', function( e ) {
                if ( 13 === e.which || 32 === e.which ) { // Enter or Space
                    e.preventDefault();
                    $( this )[0].click();
                }
            });
        },

        /**
         * Handle arrow key navigation in menus
         */
        handleMenuKeyNavigation: function( e ) {
            const $current = $( e.currentTarget );
            const $menuItems = $( '.wp-block-navigation a' );
            const currentIndex = $menuItems.index( $current );

            switch ( e.which ) {
                case 40: // ArrowDown
                case 39: // ArrowRight
                {
                    e.preventDefault();
                    const nextIndex = ( currentIndex + 1 ) % $menuItems.length;
                    $menuItems.eq( nextIndex ).focus();
                    break;
                }
                case 38: // ArrowUp
                case 37: // ArrowLeft
                {
                    e.preventDefault();
                    const prevIndex = 0 < currentIndex ? currentIndex - 1 : $menuItems.length - 1;
                    $menuItems.eq( prevIndex ).focus();
                    break;
                }
                case 36: // Home
                    e.preventDefault();
                    $menuItems.first().focus();
                    break;
                case 35: // End
                    e.preventDefault();
                    $menuItems.last().focus();
                    break;
            }
        },

        /**
         * Setup status messages for screen readers
         */
        setupStatusMessages: function() {

            // Create ARIA live region for status messages
            if ( ! $( '#rms-status-messages' ).length ) {
                $( '<div>' )
                    .attr({
                        'id': 'rms-status-messages',
                        'aria-live': 'polite',
                        'aria-atomic': 'true'
                    })
                    .addClass( 'screen-reader-text' )
                    .css({
                        'position': 'absolute',
                        'left': '-9999px',
                        'width': '1px',
                        'height': '1px',
                        'overflow': 'hidden'
                    })
                    .appendTo( 'body' );
            }
        },

        /**
         * Setup text spacing for accessibility
         */
        setupTextSpacing: function() {

            // Allow user-controlled text spacing without breaking layout
            const style = document.createElement( 'style' );
            style.textContent = `
                .user-text-spacing * {
                    line-height: 1.5 !important;
                    letter-spacing: 0.12em !important;
                    word-spacing: 0.16em !important;
                    paragraph-spacing: 2em !important;
                }
            `;
            document.head.appendChild( style );
        },

        /**
         * Reset main content spacing (no sticky header needed)
         */
        adjustMainContentSpacing: function() {

            // Reset any previous spacing adjustments for normal layout
            const $main = $( 'main' );
            if ( $main.length ) {
                $main.css({
                    'padding-top': '',
                    'margin-top': ''
                });
                console.log( 'Main content spacing reset for non-sticky layout' );
            }
        },

        /**
         * Announce message to screen readers
         */
        announceMessage: function( message ) {
            const $statusRegion = $( '#rms-status-messages' );
            $statusRegion.text( message );

            // Clear message after 5 seconds
            setTimeout( function() {
                $statusRegion.empty();
            }, 5000 );
        },

        /**
         * Setup navigation functionality
         */
        setupNavigation: function() {

            // Set initial mobile state
            this.handleResize();

            // Add ARIA attributes to navigation
            $( '.wp-block-navigation' ).attr( 'role', 'navigation' );
            $( '.wp-block-navigation ul' ).attr( 'role', 'menubar' );
            $( '.wp-block-navigation-item' ).attr( 'role', 'none' );
            $( '.wp-block-navigation-item a' ).attr( 'role', 'menuitem' );
        },

        /**
         * Setup lazy loading for images
         */
        setupLazyLoading: function() {

            // Use Intersection Observer if available, fallback to scroll events
            if ( 'IntersectionObserver' in window ) {
                this.setupIntersectionObserver();
            } else {
                this.setupScrollBasedLazyLoading();
            }
        },

        /**
         * Setup count-up counters for stats section
         */
        setupCounters: function() {
            const counters = document.querySelectorAll( '.rms-counter[data-target]' );
            if ( 0 === counters.length ) {
                return;
            }

            // Respect reduced motion
            const prefersReduced = window.matchMedia( '(prefers-reduced-motion: reduce)' ).matches;

            const animateCounter = function( el ) {
                const end = parseInt( el.getAttribute( 'data-target' ), 10 ) || 0;
                const suffix = el.getAttribute( 'data-suffix' ) || '';
                if ( prefersReduced ) {
                    el.textContent = end.toLocaleString() + suffix;
                    return;
                }

                const duration = 1600; // ms
                const startTime = performance.now();
                const startVal = 0;

                const step = function( now ) {
                    const progress = Math.min( ( now - startTime ) / duration, 1 );
                    const eased = 1 - Math.pow( 1 - progress, 3 ); // easeOutCubic
                    const current = Math.floor( startVal + ( end - startVal ) * eased );
                    el.textContent = current.toLocaleString() + ( 1 === progress ? suffix : '' );
                    if ( 1 > progress ) {
                        requestAnimationFrame( step );
                    }
                };

                requestAnimationFrame( step );
            };

            // Use IntersectionObserver to trigger when visible
            if ( 'IntersectionObserver' in window ) {
                const io = new IntersectionObserver( function( entries, observer ) {
                    entries.forEach( function( entry ) {
                        if ( entry.isIntersecting ) {
                            animateCounter( entry.target );
                            observer.unobserve( entry.target );
                        }
                    });
                }, { threshold: 0.3 });

                counters.forEach( function( el ) {
                    io.observe( el );
                });
            } else {

                // Fallback: animate immediately
                counters.forEach( function( el ) {
                    animateCounter( el );
                });
            }
        },

        /**
         * Setup Intersection Observer for lazy loading
         */
        setupIntersectionObserver: function() {
            const imageObserver = new IntersectionObserver( function( entries, observer ) {
                entries.forEach( function( entry ) {
                    if ( entry.isIntersecting ) {
                        const img = entry.target;
                        img.src = img.dataset.src;
                        img.classList.remove( 'lazy' );
                        img.classList.add( 'loaded' );
                        observer.unobserve( img );
                    }
                });
            }, {
                rootMargin: '50px 0px',
                threshold: 0.01
            });

            $( '.lazy[data-src]' ).each( function() {
                imageObserver.observe( this );
            });
        },

        /**
         * Fallback scroll-based lazy loading
         */
        setupScrollBasedLazyLoading: function() {
            const self = this;
            $( window ).on( 'scroll resize', function() {
                self.loadVisibleImages();
            });
            this.loadVisibleImages(); // Load initially visible images
        },

        /**
         * Load images that are visible in viewport
         */
        loadVisibleImages: function() {
            const windowTop = $( window ).scrollTop();
            const windowBottom = windowTop + $( window ).height();

            $( '.lazy[data-src]' ).each( function() {
                const $img = $( this );
                const imgTop = $img.offset().top;

                if ( imgTop <= windowBottom + 50 ) { // 50px buffer
                    $img.attr( 'src', $img.data( 'src' ) );
                    $img.removeClass( 'lazy' ).addClass( 'loaded' );
                    $img.removeAttr( 'data-src' );
                }
            });
        },

        /**
         * Open mobile navigation
         */
        openMobileNav: function() {
            const $nav = $( '.wp-block-navigation' );
            $nav.addClass( 'is-menu-open' );
            $( 'body' ).addClass( 'mobile-menu-open' );

            // Focus management
            setTimeout( function() {
                $( '.wp-block-navigation a' ).first().focus();
            }, 100 );

            // Announce to screen readers
            this.announceMessage( 'Navigation menu opened' );
        },

        /**
         * Close mobile navigation
         */
        closeMobileNav: function() {
            const $nav = $( '.wp-block-navigation' );
            $nav.removeClass( 'is-menu-open' );
            $( 'body' ).removeClass( 'mobile-menu-open' );

            // Return focus to toggle button
            $( '.wp-block-navigation__responsive-container-open' ).focus();

            // Announce to screen readers
            this.announceMessage( 'Navigation menu closed' );
        },

        /**
         * Update skip link target based on page structure
         */
        updateSkipLinkTarget: function() {
            const $main = $( 'main, #main, [role="main"]' ).first();
            if ( $main.length && ! $main.attr( 'id' ) ) {
                $main.attr( 'id', 'main' );
            }
        },

        /**
         * Setup newsletter form with enhanced feedback
         */
        setupNewsletter: function() {
            const $form = $( '[data-newsletter-form]' );
            if ( 0 === $form.length ) {
                return;
            }

            const $input = $form.find( 'input[type="email"]' );
            const $button = $form.find( 'button[type="submit"]' );
            const $feedback = $form.find( '#newsletter-feedback' );
            const $buttonText = $button.find( '.button-text' );
            const $spinner = $button.find( '.loading-spinner' );

            const self = this;
            $form.on( 'submit', function( e ) {
                e.preventDefault();

                const email = $input.val().trim();

                // Basic email validation
                if ( ! email || ! self.isValidEmail( email ) ) {
                    self.showNewsletterFeedback( $feedback, 'error', 'Please enter a valid email address.' );
                    return;
                }

                // Show loading state
                $button.prop( 'disabled', true );
                $buttonText.hide();
                $spinner.show();

                // Simulate API call (replace with actual newsletter service)
                setTimeout( function() {

                    // Reset button state
                    $button.prop( 'disabled', false );
                    $buttonText.show();
                    $spinner.hide();

                    // Show success message
                    self.showNewsletterFeedback( $feedback, 'success', 'Thank you! You\'ve been subscribed to our newsletter.' );
                    $input.val( '' );
                }, 1500 );
            });
        },

        /**
         * Validate email format
         */
        isValidEmail: function( email ) {
            const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
            return emailRegex.test( email );
        },

        /**
         * Show newsletter feedback message
         */
        showNewsletterFeedback: function( $feedback, type, message ) {
            $feedback
                .removeClass( 'success error show' )
                .addClass( type )
                .text( message );

            setTimeout( function() {
                $feedback.addClass( 'show' );
            }, 10 );

            // Auto-hide after 5 seconds
            setTimeout( function() {
                $feedback.removeClass( 'show' );
            }, 5000 );
        }
    };

    /**
     * Initialize when DOM is ready
     */
    $( document ).ready( function() {
        RMSTheme.init();
        
        // Fix flag data URIs for language switcher
        const flagImages = document.querySelectorAll( '.flag-icon img' );
        flagImages.forEach( function( img ) {
            const src = img.getAttribute( 'src' );
            if ( src && src.startsWith( 'image/' ) && src.includes( 'base64,' ) ) {
                const fixedSrc = 'data:' + src;
                img.setAttribute( 'src', fixedSrc );
                img.style.display = 'inline-block';
            }
        });
    });

    /**
     * Make RMSTheme available globally for debugging and extending
     */
    window.RMSTheme = RMSTheme;

}( jQuery ) );
