/**
 * Polylang Language Switcher Frontend JavaScript
 * Handles dropdown functionality with full accessibility support
 */

( function() {
	'use strict';

	/**
	 * Initialize language switcher dropdowns when DOM is ready
	 */
	document.addEventListener( 'DOMContentLoaded', function() {
		initializeLanguageSwitchers();
	});

	/**
	 * Initialize all language switcher components
	 */
	function initializeLanguageSwitchers() {
		const switchers = document.querySelectorAll( '.wp-block-rms-polylang-language-switcher' );

		if ( ! switchers.length ) {
			return;
		}

		switchers.forEach( function( switcher ) {
			const button = switcher.querySelector( '.language-switcher-button' );
			const menu = switcher.querySelector( '.language-switcher-menu' );

			if ( ! button || ! menu ) {
				return;
			}

			// Initialize dropdown functionality
			initializeDropdown( button, menu, switcher );
		});
	}

	/**
	 * Initialize individual dropdown
	 *
	 * @param {HTMLButtonElement} button - The dropdown button
	 * @param {HTMLUListElement} menu - The dropdown menu
	 * @param {HTMLElement} container - The container element
	 */
	function initializeDropdown( button, menu, container ) {
		let isOpen = false;
		let focusedIndex = -1;
		const menuItems = Array.from( menu.querySelectorAll( '.language-link' ) );

		// Button click handler
		button.addEventListener( 'click', function( e ) {
			e.preventDefault();
			e.stopPropagation();
			toggleDropdown();
		});

		// Keyboard navigation for button
		button.addEventListener( 'keydown', function( e ) {
			switch ( e.key ) {
				case 'ArrowDown':
				case 'ArrowUp':
					e.preventDefault();
					if ( ! isOpen ) {
						openDropdown();
					}

					// Focus first item when opening with arrow keys
					if ( 0 < menuItems.length ) {
						focusedIndex = 0;
						menuItems[0].focus();
					}
					break;
				case 'Escape':
					if ( isOpen ) {
						e.preventDefault();
						closeDropdown();
						button.focus();
					}
					break;
				case 'Enter':
				case ' ':
					e.preventDefault();
					toggleDropdown();
					break;
			}
		});

		// Menu keyboard navigation
		menu.addEventListener( 'keydown', function( e ) {
			switch ( e.key ) {
				case 'ArrowDown':
					e.preventDefault();
					focusedIndex = Math.min( focusedIndex + 1, menuItems.length - 1 );
					menuItems[focusedIndex].focus();
					break;
				case 'ArrowUp':
					e.preventDefault();
					focusedIndex = Math.max( focusedIndex - 1, 0 );
					menuItems[focusedIndex].focus();
					break;
				case 'Home':
					e.preventDefault();
					focusedIndex = 0;
					menuItems[0].focus();
					break;
				case 'End':
					e.preventDefault();
					focusedIndex = menuItems.length - 1;
					menuItems[focusedIndex].focus();
					break;
				case 'Escape':
					e.preventDefault();
					closeDropdown();
					button.focus();
					break;
				case 'Tab':

					// Allow natural tab behavior but close dropdown
					closeDropdown();
					break;
			}
		});

		// Click outside to close
		document.addEventListener( 'click', function( e ) {
			if ( ! container.contains( e.target ) && isOpen ) {
				closeDropdown();
			}
		});

		// Focus management for menu items
		menuItems.forEach( function( item, index ) {
			item.addEventListener( 'focus', function() {
				focusedIndex = index;
			});

			item.addEventListener( 'click', function() {

				// Let the link work naturally, but close dropdown first
				closeDropdown();
			});
		});

		/**
		 * Toggle dropdown open/closed state
		 */
		function toggleDropdown() {
			if ( isOpen ) {
				closeDropdown();
			} else {
				openDropdown();
			}
		}

		/**
		 * Open dropdown menu
		 */
		function openDropdown() {
			isOpen = true;
			button.setAttribute( 'aria-expanded', 'true' );
			menu.removeAttribute( 'hidden' );

			// Add class for any CSS animations
			container.classList.add( 'is-open' );

			// Position dropdown if needed (for mobile responsive)
			positionDropdown();

			// Announce to screen readers
			announceToScreenReader( 'Language menu opened' );
		}

		/**
		 * Close dropdown menu
		 */
		function closeDropdown() {
			isOpen = false;
			button.setAttribute( 'aria-expanded', 'false' );
			menu.setAttribute( 'hidden', '' );
			focusedIndex = -1;

			// Remove class for any CSS animations
			container.classList.remove( 'is-open' );

			// Announce to screen readers
			announceToScreenReader( 'Language menu closed' );
		}

		/**
		 * Position dropdown for responsive design
		 */
		function positionDropdown() {
			const rect = container.getBoundingClientRect();
			const viewportHeight = window.innerHeight;
			const viewportWidth = window.innerWidth;

			// Reset any previous positioning
			menu.style.removeProperty( 'top' );
			menu.style.removeProperty( 'bottom' );
			menu.style.removeProperty( 'left' );
			menu.style.removeProperty( 'right' );
			menu.style.removeProperty( 'transform' );

			// Check if dropdown would go off-screen
			const menuRect = menu.getBoundingClientRect();

			// Horizontal positioning
			if ( rect.right > viewportWidth - 20 ) {
				menu.style.right = '0';
				menu.style.left = 'auto';
			}

			// Vertical positioning
			if ( rect.bottom + menuRect.height > viewportHeight - 20 ) {
				menu.style.bottom = '100%';
				menu.style.top = 'auto';
				menu.style.marginBottom = '2px';
				menu.style.marginTop = '0';
			}
		}

		/**
		 * Announce messages to screen readers
		 *
		 * @param {string} message - Message to announce
		 */
		function announceToScreenReader( message ) {
			const announcement = document.createElement( 'div' );
			announcement.textContent = message;
			announcement.setAttribute( 'aria-live', 'polite' );
			announcement.setAttribute( 'aria-atomic', 'true' );
			announcement.style.position = 'absolute';
			announcement.style.left = '-10000px';
			announcement.style.width = '1px';
			announcement.style.height = '1px';
			announcement.style.overflow = 'hidden';

			document.body.appendChild( announcement );

			// Remove announcement after it's been read
			setTimeout( function() {
				if ( announcement.parentNode ) {
					announcement.parentNode.removeChild( announcement );
				}
			}, 1000 );
		}

		// Handle window resize for responsive positioning
		let resizeTimeout;
		window.addEventListener( 'resize', function() {
			if ( isOpen ) {
				clearTimeout( resizeTimeout );
				resizeTimeout = setTimeout( function() {
					positionDropdown();
				}, 100 );
			}
		});

		// Handle escape key globally
		document.addEventListener( 'keydown', function( e ) {
			if ( 'Escape' === e.key && isOpen ) {
				closeDropdown();
				button.focus();
			}
		});
	}

	/**
	 * Handle reduced-motion preferences
	 */
	function handleReducedMotion() {
		const prefersReducedMotion = window.matchMedia( '(prefers-reduced-motion: reduce)' );

		if ( prefersReducedMotion.matches ) {
			document.documentElement.style.setProperty( '--animation-duration', '0ms' );
			document.documentElement.style.setProperty( '--transition-duration', '0ms' );
		}

		// Listen for changes
		prefersReducedMotion.addEventListener( 'change', function() {
			if ( prefersReducedMotion.matches ) {
				document.documentElement.style.setProperty( '--animation-duration', '0ms' );
				document.documentElement.style.setProperty( '--transition-duration', '0ms' );
			} else {
				document.documentElement.style.removeProperty( '--animation-duration' );
				document.documentElement.style.removeProperty( '--transition-duration' );
			}
		});
	}

	// Initialize reduced motion handling
	handleReducedMotion();

}() );
