<?php
/**
 * RMS Block Theme functions and definitions
 *
 * @package RMS
 */

// Prevent direct access.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Sets up theme defaults and registers support for various WordPress features.
 * Includes WCAG accessibility compliance features.
 */
function rms_setup() {
	// WCAG 3.1.1 - Language of Page support.
	load_theme_textdomain( 'rms', get_template_directory() . '/languages' );

	// Add support for block styles.
	add_theme_support( 'wp-block-styles' );

/**
 * One-time cleanup: remove DB-stored header template part so the file-based
 * block template (parts/header.html) is used. This helps keep nav driven by
 * blocks rather than DB overrides created via the Site Editor.
 */
add_action(
	'init',
	function () {
		if ( get_option( 'rms_cleared_header_template_part' ) ) {
			return;
		}

		$theme_slug = wp_get_theme()->get_stylesheet();
		// Delete wp_template_part named 'header' for current theme.
		$parts = get_posts(
			array(
				'post_type'      => 'wp_template_part',
				'name'           => 'header',
				'posts_per_page' => -1,
				'tax_query'      => array(
					array(
						'taxonomy' => 'wp_theme',
						'field'    => 'slug',
						'terms'    => array( $theme_slug ),
					),
				),
			)
		);
		if ( $parts ) {
			foreach ( $parts as $p ) {
				wp_delete_post( $p->ID, true );
			}
		}

		update_option( 'rms_cleared_header_template_part', 1 );
	}
);

/**
 * One-time cleanup: remove DB-stored footer template part so the file-based
 * block template (parts/footer.html) is used. This prevents drift on hosted
 * sites where the footer was customized in the Site Editor.
 */
add_action(
	'init',
	function () {
		if ( get_option( 'rms_cleared_footer_template_part' ) ) {
			return;
		}

		$theme_slug = wp_get_theme()->get_stylesheet();
		$parts      = get_posts(
			array(
				'post_type'      => 'wp_template_part',
				'name'           => 'footer',
				'posts_per_page' => -1,
				'tax_query'      => array(
					array(
						'taxonomy' => 'wp_theme',
						'field'    => 'slug',
						'terms'    => array( $theme_slug ),
					),
				),
			)
		);
		if ( $parts ) {
			foreach ( $parts as $p ) {
				wp_delete_post( $p->ID, true );
			}
		}

		update_option( 'rms_cleared_footer_template_part', 1 );
	}
);

	// Add support for full and wide align images.
	add_theme_support( 'align-wide' );

	// Add support for editor styles.
	add_theme_support( 'editor-styles' );

	// Enqueue editor styles - Load main theme styles first, then editor-specific styles.
	add_editor_style( 'style.css' );
	add_editor_style( 'editor-style.css' );

	// Add support for responsive embedded content.
	add_theme_support( 'responsive-embeds' );

	// Block theme template support.
	add_theme_support( 'block-templates' );
	add_theme_support( 'block-template-parts' );

	// WCAG 1.1.1 - Support for post thumbnails with proper alt text.
	add_theme_support( 'post-thumbnails' );

	// WCAG 4.1.2 - HTML5 semantic markup support.
	add_theme_support(
		'html5',
		array(
			'search-form',
			'comment-form',
			'comment-list',
			'gallery',
			'caption',
			'script',
			'style',
			'navigation-widgets',
		)
	);

	// Add support for custom logo.
	add_theme_support(
		'custom-logo',
		array(
			'height'      => 250,
			'width'       => 250,
			'flex-width'  => true,
			'flex-height' => true,
		)
	);

	// Add support for document title tag.
	add_theme_support( 'title-tag' );

	// Add support for automatic feed links.
	add_theme_support( 'automatic-feed-links' );
}
add_action( 'after_setup_theme', 'rms_setup' );

// Initialize RMS Theme Updater when a Worker URL is configured.
add_action(
	'after_setup_theme',
	function () {
		if ( defined( 'RMS_UPDATER_URL' ) && RMS_UPDATER_URL ) {
			require_once get_template_directory() . '/inc/class-rms-theme-updater.php';
			new RMS_Theme_Updater( get_template(), wp_get_theme()->get( 'Version' ), RMS_UPDATER_URL );
		}
	}
);

/**
 * Enqueue styles and scripts.
 */
function rms_scripts() {
	$theme_version = wp_get_theme()->get( 'Version' );
	$theme_uri     = get_template_directory_uri();

	// Enqueue main theme stylesheet.
	wp_enqueue_style(
		'rms-style',
		get_stylesheet_uri(),
		array(),
		$theme_version
	);

	// Enqueue modular CSS files.
	wp_enqueue_style(
		'rms-layout',
		$theme_uri . '/assets/css/layout.css',
		array( 'rms-style' ),
		$theme_version
	);

	wp_enqueue_style(
		'rms-components',
		$theme_uri . '/assets/css/components.css',
		array( 'rms-style' ),
		$theme_version
	);

	wp_enqueue_style(
		'rms-forms',
		$theme_uri . '/assets/css/forms.css',
		array( 'rms-style' ),
		$theme_version
	);

	// Conditionally enqueue Polylang Language Switcher frontend JavaScript.
	// Only load if Polylang is active and we're not in admin.
	if ( function_exists( 'pll_the_languages' ) && ! is_admin() ) {
		wp_enqueue_script(
			'rms-polylang-language-switcher',
			get_template_directory_uri() . '/assets/js/polylang-language-switcher.js',
			array(),
			wp_get_theme()->get( 'Version' ),
			true
		);
	}

	// Front-end counter animation script (lightweight, no deps).
	wp_enqueue_script(
		'rms-counters',
		get_template_directory_uri() . '/assets/js/counters.js',
		array(),
		wp_get_theme()->get( 'Version' ),
		true
	);
}
add_action( 'wp_enqueue_scripts', 'rms_scripts' );

/**
 * Ensure theme form styles take precedence over plugin defaults.
 * Dequeue plugin public.css to avoid conflicting base styles.
 */
function rms_dequeue_conflicting_plugin_assets() {
	// Runs after default enqueue to allow dequeue.
	wp_dequeue_style( 'rms-cv-public' );
	wp_deregister_style( 'rms-cv-public' );
}
add_action( 'wp_enqueue_scripts', 'rms_dequeue_conflicting_plugin_assets', 20 );
// Catch styles enqueued late via shortcodes (printed in head or footer).
add_action( 'wp_print_styles', 'rms_dequeue_conflicting_plugin_assets', 100 );
add_action( 'wp_print_footer_scripts', 'rms_dequeue_conflicting_plugin_assets', 100 );

/**
 * Include multilingual content management system.
 */
// Include multilingual content management system.
$multilingual_file = get_template_directory() . '/inc/class-rms-multilingual-content.php';
if ( file_exists( $multilingual_file ) ) {
	require_once $multilingual_file;
}
// Template helper functions (wrappers around multilingual content).
$template_helpers_file = get_template_directory() . '/inc/template-helpers.php';
if ( file_exists( $template_helpers_file ) ) {
	require_once $template_helpers_file;
}

/**
 * Register custom blocks.
 */
function rms_register_blocks() {
	// Register custom block category.
	add_filter(
		'block_categories_all',
		function ( $categories ) {
			return array_merge(
				array(
					array(
						'slug'  => 'rms-blocks',
						'title' => __( 'RMS Blocks', 'rms' ),
					),
				),
				$categories
			);
		}
	);

	// Register Service Card block with icon/image support.
	register_block_type(
		'rms/service-card',
		array(
			'api_version'     => 3,
			'title'           => __( 'RMS Service Card', 'rms' ),
			'description'     => __( 'A service card with glassy black/gold styling, icon or image, title, and description.', 'rms' ),
			'category'        => 'rms-blocks',
			'keywords'        => array( 'service', 'card', 'feature', 'rms' ),
			'textdomain'      => 'rms',
			'attributes'      => array(
				'iconType'    => array(
					'type'    => 'string',
					'default' => 'emoji',
				),
				'icon'        => array(
					'type'    => 'string',
					'default' => '🚀',
				),
				'imageId'     => array(
					'type'    => 'number',
					'default' => 0,
				),
				'imageUrl'    => array(
					'type'    => 'string',
					'default' => '',
				),
				'imageAlt'    => array(
					'type'    => 'string',
					'default' => '',
				),
				'title'       => array(
					'type'    => 'string',
					'default' => 'Coming Soon',
				),
				'description' => array(
					'type'    => 'string',
					'default' => 'We\'re preparing exciting new services for you. Stay tuned!',
				),
				'linkUrl'     => array(
					'type'    => 'string',
					'default' => '',
				),
			),
			'supports'        => array(
				'html'    => false,
				'spacing' => array(
					'margin'  => true,
					'padding' => true,
				),
				'color'   => array(
					'background' => true,
					'text'       => true,
				),
			),
			'example'         => array(
				'attributes' => array(
					'iconType'    => 'emoji',
					'icon'        => '⭐',
					'title'       => 'Premium Service',
					'description' => 'High-quality service with professional support and guaranteed results.',
					'linkUrl'     => '#',
					'linkText'    => 'Get Started',
				),
			),
			'render_callback' => 'rms_render_service_card_block',
		)
	);

	// Register Polylang Language Switcher block.
	register_block_type(
		'rms/polylang-language-switcher',
		array(
			'api_version'           => 3,
			'title'                 => __( 'Language Switcher', 'rms' ),
			'description'           => __( 'A dropdown language switcher powered by Polylang with full accessibility support.', 'rms' ),
			'category'              => 'widgets',
			'keywords'              => array( 'language', 'polylang', 'translation', 'switcher', 'multilingual' ),
			'textdomain'            => 'rms',
			'attributes'            => array(
				'showFlags'           => array(
					'type'    => 'boolean',
					'default' => true,
				),
				'showNames'           => array(
					'type'    => 'boolean',
					'default' => true,
				),
				'dropdownStyle'       => array(
					'type'    => 'string',
					'default' => 'default',
				),
				'hideCurrentLanguage' => array(
					'type'    => 'boolean',
					'default' => false,
				),
				'forceHome'           => array(
					'type'    => 'boolean',
					'default' => false,
				),
				'displayCurrentFirst' => array(
					'type'    => 'boolean',
					'default' => true,
				),
			),
			'supports'              => array(
				'html'       => false,
				'align'      => array( 'left', 'center', 'right' ),
				'spacing'    => array(
					'margin'  => true,
					'padding' => true,
				),
				'color'      => array(
					'text'       => true,
					'background' => true,
					'link'       => true,
				),
				'typography' => array(
					'fontSize'   => true,
					'lineHeight' => true,
				),
			),
			'example'               => array(
				'attributes' => array(
					'showFlags'     => true,
					'showNames'     => true,
					'dropdownStyle' => 'default',
				),
			),
			'editor_script_handles' => array( 'rms-polylang-language-switcher-block' ),
			'render_callback'       => 'rms_render_polylang_language_switcher_block',
		)
	);

		// Register Verify Certificate Button block.
		register_block_type(
			'rms/verify-button',
			array(
				'api_version'     => 3,
				'title'           => __( 'Verify Certificate Button', 'rms' ),
				'description'     => __( 'A localized Verify Certificate button for the header or anywhere.', 'rms' ),
				'category'        => 'widgets',
				'keywords'        => array( 'verify', 'certificate', 'button', 'rms' ),
				'attributes'      => array(
					'label'     => array( 'type' => 'string', 'default' => '' ),
					'url'       => array( 'type' => 'string', 'default' => '' ),
					'usePrimary' => array( 'type' => 'boolean', 'default' => true ),
				),
				'supports'        => array(
					'html'  => false,
					'align' => array( 'left', 'center', 'right' ),
				),
				'render_callback' => 'rms_render_verify_button_block',
			)
		);

		// Register Counter block.
		register_block_type(
			'rms/counter',
			array(
				'api_version'     => 3,
				'title'           => __( 'RMS Counter', 'rms' ),
				'description'     => __( 'Animated numeric counter with accessible labeling.', 'rms' ),
				'category'        => 'rms-blocks',
				'keywords'        => array( 'counter', 'stats', 'kpi', 'rms' ),
				'attributes'      => array(
					'value'        => array( 'type' => 'number', 'default' => 100 ),
					'prefix'       => array( 'type' => 'string', 'default' => '' ),
					'suffix'       => array( 'type' => 'string', 'default' => '' ),
					'label'        => array( 'type' => 'string', 'default' => '' ),
					'labelKey'     => array( 'type' => 'string', 'default' => '' ),
					'duration'     => array( 'type' => 'number', 'default' => 1200 ),
					'grouping'     => array( 'type' => 'boolean', 'default' => true ),
					'reduceMotion' => array( 'type' => 'boolean', 'default' => true ),
				),
				'supports'        => array(
					'html'       => false,
					'align'      => array( 'left', 'center', 'right', 'wide', 'full' ),
					'spacing'    => array( 'margin' => true, 'padding' => true ),
					'typography' => array( 'fontSize' => true ),
				),
				'editor_script_handles' => array( 'rms-counter-block' ),
				'render_callback' => 'rms_render_counter_block',
			)
		);
}
add_action( 'init', 'rms_register_blocks' );

// Do not auto-append the old header Verify button next to the language switcher anymore.
add_filter( 'rms_show_verify_button_in_switcher', '__return_false' );

/**
 * Register Stats Counter pattern programmatically to ensure availability.
 */
function rms_register_stats_pattern() {
	if ( function_exists( 'register_block_pattern' ) ) {
		// Build localized content similar to patterns/stats-counter.php
		$lang        = function_exists( 'pll_current_language' ) ? pll_current_language() : 'en';
		$stats_title = class_exists( 'RMS_Multilingual_Content' ) ? RMS_Multilingual_Content::get_content( 'stats_title', $lang ) : __( 'Our Impact in Numbers', 'rms' );
		$items       = class_exists( 'RMS_Multilingual_Content' ) ? RMS_Multilingual_Content::get_content( 'stats_items', $lang ) : array();

		if ( empty( $items ) || ! is_array( $items ) ) {
			$items = array(
				array( 'label' => __( 'Trainees', 'rms' ), 'value' => 12000, 'suffix' => '+' ),
				array( 'label' => __( 'Trainers', 'rms' ), 'value' => 350, 'suffix' => '+' ),
				array( 'label' => __( 'International Partners', 'rms' ), 'value' => 25, 'suffix' => '+' ),
				array( 'label' => __( 'Excellence Achieved', 'rms' ), 'value' => 98, 'suffix' => '%' ),
			);
		}

		$columns_markup = '';
		foreach ( $items as $item ) {
			$value  = isset( $item['value'] ) ? (int) $item['value'] : 0;
			$label  = isset( $item['label'] ) ? $item['label'] : '';
			$suffix = isset( $item['suffix'] ) ? $item['suffix'] : '';

			$block_json = wp_json_encode(
				array(
					'value'    => $value,
					'suffix'   => (string) $suffix,
					'label'    => (string) $label,
					'duration' => 1500,
				)
			);

			$columns_markup .= "\n<!-- wp:column {\"className\":\"rms-stat\"} -->\n<div class=\"wp-block-column rms-stat\">\n\t<!-- wp:rms/counter $block_json /-->\n</div>\n<!-- /wp:column -->";
		}

		$pattern_markup = '<!-- wp:group {"align":"full","style":{"spacing":{"padding":{"top":"3rem","bottom":"3rem"}}},"backgroundColor":"primary-dark","className":"rms-stats"} -->'
			. '<div class="wp-block-group alignfull has-primary-dark-background-color has-background rms-stats" style="padding-top:3rem;padding-bottom:3rem">'
			. '<!-- wp:group {"align":"wide","layout":{"type":"constrained"}} -->'
			. '<div class="wp-block-group alignwide">'
			. '<!-- wp:heading {"textAlign":"center","level":2,"style":{"typography":{"fontWeight":"800"}}} -->'
			. '<h2 class="wp-block-heading has-text-align-center">' . esc_html( $stats_title ) . '</h2>'
			. '<!-- /wp:heading -->'
			. '<!-- wp:spacer {"height":"20px"} --><div style="height:20px" aria-hidden="true" class="wp-block-spacer"></div><!-- /wp:spacer -->'
			. '<!-- wp:columns {"align":"wide","style":{"spacing":{"blockGap":"2rem"}}} -->'
			. '<div class="wp-block-columns alignwide">' . $columns_markup . '</div>'
			. '<!-- /wp:columns -->'
			. '</div>'
			. '<!-- /wp:group -->'
			. '</div>'
			. '<!-- /wp:group -->';

		register_block_pattern(
			'rms/stats-counter',
			array(
				'title'       => __( 'Stats Counter', 'rms' ),
				'categories'  => array( 'featured', 'text' ),
				'description' => __( 'Accessible, localized stats/counter section with four items and count-up animation.', 'rms' ),
				'content'     => $pattern_markup,
			)
		);
	}
}
add_action( 'init', 'rms_register_stats_pattern', 20 );

/**
 * Register a localized Services Grid pattern programmatically.
 * This ensures service cards (titles/descriptions/links) are rendered in the current locale.
 */
function rms_register_services_pattern() {
	if ( function_exists( 'register_block_pattern' ) ) {
		$lang          = function_exists( 'pll_current_language' ) ? pll_current_language() : 'en';
		$title         = class_exists( 'RMS_Multilingual_Content' ) ? RMS_Multilingual_Content::get_content( 'services_title', $lang ) : __( 'Our Services', 'rms' );
		$subtitle      = class_exists( 'RMS_Multilingual_Content' ) ? RMS_Multilingual_Content::get_content( 'services_subtitle', $lang ) : __( 'Professional solutions tailored to your needs', 'rms' );
		$service_cards = class_exists( 'RMS_Multilingual_Content' ) ? RMS_Multilingual_Content::get_service_cards( $lang ) : array();

		// Fallback set if none provided.
		if ( empty( $service_cards ) ) {
			$service_cards = array(
				array(
					'icon'        => '\ud83d\ude80',
					'title'       => __( 'Innovation', 'rms' ),
					'description' => __( 'Cutting-edge solutions powered by the latest technology and creative thinking.', 'rms' ),
					'link'        => '#',
				),
				array(
					'icon'        => '\ud83d\udcbc',
					'title'       => __( 'Business Solutions', 'rms' ),
					'description' => __( 'Comprehensive business strategies to help your company grow and succeed.', 'rms' ),
					'link'        => '#',
				),
				array(
					'icon'        => '\ud83c\udfaf',
					'title'       => __( 'Consulting', 'rms' ),
					'description' => __( 'Expert advice and guidance to navigate complex challenges and opportunities.', 'rms' ),
					'link'        => '#',
				),
			);
		}

		$columns_markup = '';
		foreach ( $service_cards as $card ) {
			$attrs = array(
				'iconType'    => 'emoji',
				'icon'        => (string) ( $card['icon'] ?? '' ),
				'title'       => (string) ( $card['title'] ?? '' ),
				'description' => (string) ( $card['description'] ?? '' ),
				'linkUrl'     => (string) ( $card['link'] ?? '' ),
			);
			$block_json = wp_json_encode( $attrs );
			$columns_markup .= "\n<!-- wp:column -->\n<div class=\"wp-block-column\">\n\t<!-- wp:rms/service-card $block_json /-->\n</div>\n<!-- /wp:column -->";
		}

		$pattern_markup = '<!-- wp:group {"align":"full","style":{"spacing":{"padding":{"top":"4rem","bottom":"4rem"}}}} -->'
			. '<div class="wp-block-group alignfull" style="padding-top:4rem;padding-bottom:4rem">'
			. '<!-- wp:heading {"textAlign":"center","level":2} -->'
			. '<h2 class="wp-block-heading has-text-align-center">' . esc_html( $title ) . '</h2>'
			. '<!-- /wp:heading -->'
			. '<!-- wp:paragraph {"align":"center"} -->'
			. '<p class="has-text-align-center">' . esc_html( $subtitle ) . '</p>'
			. '<!-- /wp:paragraph -->'
			. '<!-- wp:columns {"align":"wide"} -->'
			. '<div class="wp-block-columns alignwide">' . $columns_markup . '</div>'
			. '<!-- /wp:columns -->'
			. '</div>'
			. '<!-- /wp:group -->';

		register_block_pattern(
			'rms/services-grid',
			array(
				'title'       => __( 'Services Grid', 'rms' ),
				'categories'  => array( 'featured', 'text' ),
				'description' => __( 'Localized services grid using RMS Service Cards.', 'rms' ),
				'content'     => $pattern_markup,
			)
		);
	}
}
add_action( 'init', 'rms_register_services_pattern', 20 );

/**
 * Register a reusable 'Justified' block style for paragraphs and lists.
 */
function rms_register_justified_block_style() {
	if ( function_exists( 'register_block_style' ) ) {
		$targets = array( 'core/paragraph', 'core/list' );
		foreach ( $targets as $block ) {
			register_block_style(
				$block,
				array(
					'name'  => 'justified',
					'label' => __( 'Justified', 'rms' ),
				)
			);
		}
	}
}
add_action( 'init', 'rms_register_justified_block_style' );

/**
 * Ensure block theme template parts are properly recognized.
 */
function rms_register_block_template_parts() {
	// Ensure template parts directory is recognized.
	add_filter( 'get_block_templates', 'rms_add_block_template_parts', 10, 3 );
}
add_action( 'init', 'rms_register_block_template_parts' );

/**
 * Add custom template parts to block templates.
 *
 * @param array  $templates     Existing templates.
 * @param array  $query         Query args.
 * @param string $template_type Template type.
 * @return array Modified templates.
 */
function rms_add_block_template_parts( $templates, $query, $template_type ) {
	if ( 'wp_template_part' === $template_type ) {
		$template_parts_dir = get_template_directory() . '/parts/';

		if ( is_dir( $template_parts_dir ) ) {
			$part_files = glob( $template_parts_dir . '*.html' );
			foreach ( $part_files as $part_file ) {
				$slug = basename( $part_file, '.html' );

				// Check if this template part already exists.
				$exists = false;
				foreach ( $templates as $template ) {
					if ( $template->slug === $slug ) {
						$exists = true;
						break;
					}
				}

				if ( ! $exists ) {
					$template                 = new WP_Block_Template();
					$template->slug           = $slug;
					$template->id             = get_stylesheet() . '//' . $slug;
					$template->theme          = get_stylesheet();
					$template->content        = file_get_contents( $part_file ); // phpcs:ignore WordPress.WP.AlternativeFunctions.file_get_contents_file_get_contents
					$template->source         = 'theme';
					$template->type           = 'wp_template_part';
					$template->title          = ucfirst( $slug );
					$template->status         = 'publish';
					$template->has_theme_file = true;
					$template->is_custom      = false;
					$template->area           = ( 'header' === $slug ) ? 'header' : ( ( 'footer' === $slug ) ? 'footer' : 'uncategorized' );

					$templates[] = $template;
				}
			}
		}
	}

	return $templates;
}

/**
 * Enqueue block editor scripts conditionally.
 */
function rms_enqueue_block_editor_assets() {
	// Always enqueue Service Card block script in block editor.
	wp_enqueue_script(
		'rms-service-card-block',
		get_template_directory_uri() . '/assets/js/blocks/service-card.js',
		array( 'wp-blocks', 'wp-block-editor', 'wp-components', 'wp-i18n', 'wp-element' ),
		wp_get_theme()->get( 'Version' ) . '-' . filemtime( get_template_directory() . '/assets/js/blocks/service-card.js' ),
		true
	);

	// JS translations for Service Card block strings in the editor.
	wp_set_script_translations( 'rms-service-card-block', 'rms', get_template_directory() . '/languages' );

	// Only enqueue Polylang Language Switcher block script if Polylang is active.
	if ( function_exists( 'pll_the_languages' ) ) {
		wp_enqueue_script(
			'rms-polylang-language-switcher-block',
			get_template_directory_uri() . '/assets/js/blocks/polylang-language-switcher.js',
			array( 'wp-blocks', 'wp-block-editor', 'wp-components', 'wp-i18n', 'wp-element' ),
			wp_get_theme()->get( 'Version' ),
			true
		);
		// Ensure JS translations for the language switcher block.
		wp_set_script_translations( 'rms-polylang-language-switcher-block', 'rms', get_template_directory() . '/languages' );
	}

	// Enqueue Counter block script in the block editor.
	wp_enqueue_script(
		'rms-counter-block',
		get_template_directory_uri() . '/assets/js/blocks/counter.js',
		array( 'wp-blocks', 'wp-block-editor', 'wp-components', 'wp-i18n', 'wp-element' ),
		wp_get_theme()->get( 'Version' ) . '-' . filemtime( get_template_directory() . '/assets/js/blocks/counter.js' ),
		true
	);
	wp_set_script_translations( 'rms-counter-block', 'rms', get_template_directory() . '/languages' );
}
add_action( 'enqueue_block_editor_assets', 'rms_enqueue_block_editor_assets' );

/**
 * Security helper functions for nonce protection.
 */

// Removed header CTA shortcode and injection filter per updated requirements.


/**
 * Generate nonce field for forms.
 *
 * @param string $action The action name for the nonce.
 * @param string $name   The name attribute for the nonce field (default: 'rms_nonce').
 * @return string        The nonce field HTML.
 */
function rms_get_nonce_field( $action, $name = 'rms_nonce' ) {
	return wp_nonce_field( $action, $name, true, false );
}

/**
 * Verify nonce for form submissions.
 *
 * @param string $nonce  The nonce value to verify.
 * @param string $action The action name for the nonce.
 * @return bool          True if nonce is valid, false otherwise.
 */
function rms_verify_nonce( $nonce, $action ) {
	return wp_verify_nonce( $nonce, $action );
}

/**
 * Get nonce for AJAX requests.
 *
 * @param string $action The action name for the nonce.
 * @return string        The nonce value.
 */
function rms_create_nonce( $action ) {
	return wp_create_nonce( $action );
}

/**
 * Helper function to sanitize Service Card attributes.
 *
 * @param array $attributes Raw block attributes.
 * @return array Sanitized attributes.
 */
function rms_sanitize_service_card_attributes( $attributes ) {
	return array(
		'icon_type'   => esc_html( $attributes['iconType'] ?? 'emoji' ),
		'icon'        => esc_html( $attributes['icon'] ?? '🚀' ),
		'image_url'   => esc_url( $attributes['imageUrl'] ?? '' ),
		'image_alt'   => esc_attr( $attributes['imageAlt'] ?? '' ),
		'title'       => esc_html( $attributes['title'] ?? __( 'Coming Soon', 'rms' ) ),
		'description' => esc_html( $attributes['description'] ?? __( 'We\'re preparing exciting new services for you. Stay tuned!', 'rms' ) ),
		'link_url'    => esc_url( $attributes['linkUrl'] ?? '' ),
	);
}

/**
 * Helper function to build Service Card CSS classes.
 *
 * @param string $icon_type Icon type (emoji|image).
 * @param string $image_url Image URL.
 * @param string $link_url  Link URL.
 * @return array Classes and wrapper attributes.
 */
function rms_build_service_card_classes( $icon_type, $image_url, $link_url ) {
	$icon_classes = 'rms-card__icon';
	if ( 'image' === $icon_type && $image_url ) {
		$icon_classes .= ' rms-card__icon--has-image';
	}

	$card_classes  = 'rms-card service-card rms-card--icon-top';
	$wrapper_attrs = get_block_wrapper_attributes(
		array(
			'class' => $card_classes . ( $link_url ? ' rms-card--clickable' : '' ),
		)
	);

	return array(
		'icon_classes'  => $icon_classes,
		'wrapper_attrs' => $wrapper_attrs,
	);
}

/**
 * Helper function to render Service Card icon/image.
 *
 * @param string $icon_type   Icon type (emoji|image).
 * @param string $icon        Icon text.
 * @param string $image_url   Image URL.
 * @param string $image_alt   Image alt.
 * @param string $icon_classes Icon CSS classes.
 * @return string HTML output.
 */
function rms_render_service_card_icon( $icon_type, $icon, $image_url, $image_alt, $icon_classes ) {
	ob_start();
	?>
	<div class="<?php echo esc_attr( $icon_classes ); ?>">
		<?php if ( 'image' === $icon_type && $image_url ) : ?>
			<img src="<?php echo esc_url( $image_url ); ?>" alt="<?php echo esc_attr( $image_alt ); ?>" class="rms-card__image" />
		<?php else : ?>
			<?php echo esc_html( $icon ); ?>
		<?php endif; ?>
	</div>
	<?php
	return ob_get_clean();
}

/**
 * Helper function to render Service Card content.
 *
 * @param string $title       Title.
 * @param string $description Description.
 * @param string $content     Inner blocks content.
 * @return string HTML output.
 */
function rms_render_service_card_content( $title, $description, $content ) {
	ob_start();
	?>
	<div class="rms-card__content">
		<h3 class="rms-card__title"><?php echo esc_html( $title ); ?></h3>
		<p class="rms-card__body"><?php echo esc_html( $description ); ?></p>
		<?php
		// Output any inner blocks content inserted in the editor.
		if ( ! empty( $content ) ) {
			echo $content; // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
		}
		?>
	</div>
	<?php
	return ob_get_clean();
}

/**
 * Render Service Card block.
 *
 * @param array         $attributes Block attributes.
 * @param string        $content    Inner content.
 * @param WP_Block|null $block      Block instance.
 * @return string HTML.
 */
function rms_render_service_card_block( $attributes, $content = '', $block = null ) {
	// Sanitize attributes using helper function.
	$sanitized   = rms_sanitize_service_card_attributes( $attributes );
	$icon_type   = $sanitized['icon_type'];
	$icon        = $sanitized['icon'];
	$image_url   = $sanitized['image_url'];
	$image_alt   = $sanitized['image_alt'];
	$title       = $sanitized['title'];
	$description = $sanitized['description'];
	$link_url    = $sanitized['link_url'];

	// Render-time localization retrofit: if current language is not English, try replacing
	// known English placeholders or sample titles with localized service content.
	$lang = function_exists( 'pll_current_language' ) ? pll_current_language() : 'en';
	if ( 'en' !== $lang && class_exists( 'RMS_Multilingual_Content' ) ) {
		$should_override = false;

		$en_cards  = RMS_Multilingual_Content::get_service_cards( 'en' );
		$loc_cards = RMS_Multilingual_Content::get_service_cards( $lang );

		// Helper: normalize a URL path to a canonical services path without language prefix.
		$canonical_from_url = static function ( $url ) {
			$path = (string) wp_parse_url( $url, PHP_URL_PATH );
			if ( '' === $path ) {
				return '';
			}
			// Ensure trailing slash.
			if ( '/' !== substr( $path, -1 ) ) {
				$path .= '/';
			}
			// Strip leading language code (/fr/, /ar/) if present.
			$path_no_lang = preg_replace( '#^/(?:fr|ar)/#', '/', $path );
			// Keep only /services/.../
			if ( preg_match( '#(/services/[^\s\?\#]+/)+#', $path_no_lang, $m ) ) {
				return trailingslashit( $m[0] );
			}
			return rtrim( $path_no_lang, '/' ) . '/';
		};

		// 1) Prefer link-based mapping: match by canonical services path.
		$target_idx = null;
		if ( ! empty( $link_url ) ) {
			$canonical = $canonical_from_url( $link_url );
			if ( $canonical ) {
				foreach ( $en_cards as $idx => $card ) {
					$en_canon = $canonical_from_url( $card['link'] ?? '' );
					if ( $en_canon && $en_canon === $canonical ) {
						$target_idx = $idx;
						break;
					}
				}
			}
		}

		// 2) Known sample titles mapping (from older template content) -> approximate indices.
		if ( null === $target_idx && $title ) {
			$map = array(
				'Premium Services'   => 0, // Innovation
				'Expert Team'        => 2, // Consulting
				'Quality Guarantee'   => 3, // Technical Support
			);
			if ( isset( $map[ $title ] ) ) {
				$target_idx      = $map[ $title ];
				$should_override = true; // Strong hint this is legacy English content.
			}
		}

		// 3) Placeholder mapping for default demo text.
		$english_placeholders = array(
			'Coming Soon',
			"We're preparing exciting new services for you. Stay tuned!",
		);
		if ( null === $target_idx && ( in_array( $title, $english_placeholders, true ) || in_array( $description, $english_placeholders, true ) ) ) {
			$target_idx      = 0; // Default to first card as a safe fallback.
			$should_override = true;
		}

		// 4) Icon-based mapping (only if icon uniquely identifies a card in EN set).
		if ( null === $target_idx && ! empty( $icon ) ) {
			$matches = array();
			foreach ( $en_cards as $idx => $card ) {
				if ( isset( $card['icon'] ) && $card['icon'] === $icon ) {
					$matches[] = $idx;
				}
			}
			if ( 1 === count( $matches ) ) {
				$target_idx = $matches[0];
			}
		}

		if ( null !== $target_idx && isset( $loc_cards[ $target_idx ] ) ) {
			$loc = $loc_cards[ $target_idx ];
			// Decide whether to override: replace when incoming strings are empty or look like English/demo.
			$looks_english = ( in_array( $title, $english_placeholders, true ) || in_array( $description, $english_placeholders, true ) );
			$match_en      = false;
			if ( isset( $en_cards[ $target_idx ] ) ) {
				$en = $en_cards[ $target_idx ];
				$match_en = ( ( isset( $en['title'] ) && $title === $en['title'] ) || ( isset( $en['description'] ) && $description === $en['description'] ) );
			}
			if ( $should_override || $looks_english || $match_en || '' === $title || '' === $description ) {
				$title       = isset( $loc['title'] ) ? $loc['title'] : $title;
				$description = isset( $loc['description'] ) ? $loc['description'] : $description;
				// If link was missing, use localized link for better UX.
				if ( empty( $link_url ) && ! empty( $loc['link'] ) ) {
					$link_url = $loc['link'];
				}
			}
		}
	}

	// Build CSS classes using helper function.
	$classes = rms_build_service_card_classes( $icon_type, $image_url, $link_url );

	// Render icon and content using helper functions.
	$icon_html    = rms_render_service_card_icon( $icon_type, $icon, $image_url, $image_alt, $classes['icon_classes'] );
	$content_html = rms_render_service_card_content( $title, $description, $content );

	// Return the card HTML as string instead of echoing.
	if ( $link_url ) {
		// get_block_wrapper_attributes() returns a prepared attribute string.
		return '<a href="' . esc_url( $link_url ) . '" ' . $classes['wrapper_attrs'] . // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
			'>' . $icon_html . $content_html . '</a>';
	} else {
		// get_block_wrapper_attributes() returns a prepared attribute string.
		return '<div ' . $classes['wrapper_attrs'] . // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
			'>' . $icon_html . $content_html . '</div>';
	}
}

/**
 * Render Polylang Language Switcher block.
 *
 * @param array         $attributes Block attributes.
 * @param string        $content    Inner content (unused).
 * @param WP_Block|null $block    Block instance (unused).
 * @return string HTML.
 */
function rms_render_polylang_language_switcher_block( $attributes, $content = '', $block = null ) {
	// Check if Polylang is active.
	if ( ! function_exists( 'pll_the_languages' ) ) {
		return '<div class="wp-block-rms-polylang-language-switcher">
			<p class="polylang-not-active">' . esc_html__( 'Polylang plugin is not active.', 'rms' ) . '</p>
		</div>';
	}

	// Extract attributes with defaults.
	$show_flags            = $attributes['showFlags'] ?? true;
	$show_names            = $attributes['showNames'] ?? true;
	$dropdown_style        = $attributes['dropdownStyle'] ?? 'default';
	$hide_current          = $attributes['hideCurrentLanguage'] ?? false;
	$force_home            = $attributes['forceHome'] ?? false;
	$display_current_first = $attributes['displayCurrentFirst'] ?? true;

	// Get languages from Polylang.
	$languages = pll_the_languages(
		array(
			'dropdown'      => 0, // We'll create our own dropdown.
			'show_flags'    => $show_flags,
			'show_names'    => $show_names,
			'hide_if_empty' => true,
			'force_home'    => $force_home,
			'hide_current'  => $hide_current,
			'post_id'       => null,
			'raw'           => 1,
		)
	);

	// Note: English URL will not have /en prefix because it's the default language.
	// This is standard Polylang behavior when "Remove language code from default language URL" is enabled.

	// Provide fallback languages only when Polylang has no languages configured.
	if ( empty( $languages ) ) {
		// Provide sample languages when no languages are configured.
		$languages = array(
			array(
				'name'   => 'English',
				'slug'   => 'en',
				'locale' => 'en_US',
				'url'    => home_url( '/' ),
				'flag'   => '🇺🇸',
			),
			array(
				'name'   => 'Français',
				'slug'   => 'fr',
				'locale' => 'fr_FR',
				'url'    => home_url( '/fr/' ),
				'flag'   => '🇫🇷',
			),
		);
	}

	// Get current language.
	$current_lang     = pll_current_language();
	$current_language = null;

	// Find current language data and reorder if needed.
	if ( $display_current_first && $current_lang ) {
		foreach ( $languages as $key => $lang ) {
			if ( $lang['slug'] === $current_lang ) {
				$current_language = $lang;
				unset( $languages[ $key ] );
				$languages = array( $current_language ) + $languages;
				break;
			}
		}
	}

	// Generate unique ID for accessibility.
	$unique_id = 'language-switcher-' . wp_unique_id();
	$button_id = $unique_id . '-button';
	$menu_id   = $unique_id . '-menu';

	// Build CSS classes.
	$wrapper_classes = array(
		'wp-block-rms-polylang-language-switcher',
		'language-switcher-' . esc_attr( $dropdown_style ),
	);

	if ( $show_flags && ! $show_names ) {
		$wrapper_classes[] = 'flags-only';
	} elseif ( $show_names && ! $show_flags ) {
		$wrapper_classes[] = 'names-only';
	}

	// Get block wrapper attributes.
	$wrapper_attrs = get_block_wrapper_attributes(
		array(
			'class' => implode( ' ', $wrapper_classes ),
		)
	);

	// Start building the HTML output.
	ob_start();
	?>
	<div <?php echo $wrapper_attrs; // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped ?>>
		<?php
		// Add dropdown button (current language or first language).
		$button_language = $display_current_first && $current_language ? $current_language : reset( $languages );
		?>
		<button type="button"
			class="language-switcher-button"
			id="<?php echo esc_attr( $button_id ); ?>"
			aria-expanded="false"
			aria-haspopup="menu"
			aria-controls="<?php echo esc_attr( $menu_id ); ?>"
			aria-label="<?php echo esc_attr__( 'Choose language', 'rms' ); ?>">

			<?php if ( $show_flags && ! empty( $button_language['flag'] ) ) : ?>
				<?php
				// Fix data URI prefix for base64 images.
				$flag_content = $button_language['flag'];
				// Fix missing data: protocol in image sources.
				$flag_content = str_replace( 'src="image/png;base64,', 'src="data:image/png;base64,', $flag_content );
				$flag_content = str_replace( 'src="image/jpeg;base64,', 'src="data:image/jpeg;base64,', $flag_content );
				$flag_content = str_replace( 'src="image/jpg;base64,', 'src="data:image/jpg;base64,', $flag_content );
				$flag_content = str_replace( 'src="image/gif;base64,', 'src="data:image/gif;base64,', $flag_content );
				?>
				<span class="flag-icon" aria-hidden="true"><?php echo wp_kses_post( $flag_content ); ?></span>
			<?php endif; ?>

			<?php if ( $show_names ) : ?>
				<span class="language-name"><?php echo esc_html( $button_language['name'] ); ?></span>
			<?php endif; ?>

			<span class="dropdown-arrow" aria-hidden="true">▼</span>
		</button>

		<ul class="language-switcher-menu"
			id="<?php echo esc_attr( $menu_id ); ?>"
			role="menu"
			aria-labelledby="<?php echo esc_attr( $button_id ); ?>"
			hidden>

			<?php foreach ( $languages as $lang ) : ?>
				<?php
				$is_current   = ( $current_lang && $lang['slug'] === $current_lang );
				$item_classes = array( 'language-item' );

				if ( $is_current ) {
					$item_classes[] = 'current-language';
				}

				// Skip if current language should be hidden.
				if ( $is_current && $hide_current ) {
					continue;
				}
				?>
				<li class="<?php echo esc_attr( implode( ' ', $item_classes ) ); ?>" role="none">
					<a href="<?php echo esc_url( $lang['url'] ); ?>"
						class="language-link"
						role="menuitem"
						lang="<?php echo esc_attr( $lang['locale'] ); ?>"
						hreflang="<?php echo esc_attr( $lang['locale'] ); ?>"
						<?php
						if ( $is_current ) :
							?>
							aria-current="page" <?php endif; ?>>

						<?php if ( $show_flags && ! empty( $lang['flag'] ) ) : ?>
							<?php
							// Fix data URI prefix for base64 images.
							$flag_content = $lang['flag'];
							// Fix missing data: protocol in image sources.
							$flag_content = str_replace( 'src="image/png;base64,', 'src="data:image/png;base64,', $flag_content );
							$flag_content = str_replace( 'src="image/jpeg;base64,', 'src="data:image/jpeg;base64,', $flag_content );
							$flag_content = str_replace( 'src="image/jpg;base64,', 'src="data:image/jpg;base64,', $flag_content );
							$flag_content = str_replace( 'src="image/gif;base64,', 'src="data:image/gif;base64,', $flag_content );
							?>
							<span class="flag-icon" aria-hidden="true"><?php echo wp_kses_post( $flag_content ); ?></span>
						<?php endif; ?>

						<?php if ( $show_names ) : ?>
							<span class="language-name"><?php echo esc_html( $lang['name'] ); ?></span>
						<?php endif; ?>
					</a>
				</li>
			<?php endforeach; ?>
		</ul>
	</div>
	<?php
	// Optionally append a Verify button next to the language switcher (header utility area).
	// Default enabled when no dedicated header button block is used.
	$append_verify = apply_filters( 'rms_show_verify_button_in_switcher', true );
	$output        = ob_get_clean();
	if ( $append_verify && function_exists( 'esc_url' ) ) {
		$output .= rms_render_verify_button_block( array( 'usePrimary' => true ), '' );
	}
	return $output;
}

/**
 * WCAG Accessibility Enhancements.
 */

/**
 * Ensure images have proper alt text (WCAG 1.1.1).
 *
 * @param string $content Post content.
 * @return string Filtered content.
 */
function rms_add_missing_alt_text( $content ) {
	// Add role="img" and empty alt for decorative images without alt text.
	$content = preg_replace( '/<img(?![^>]*alt=)([^>]*)>/i', '<img$1 alt="" role="img">', $content );
	return $content;
}
add_filter( 'the_content', 'rms_add_missing_alt_text' );

/**
 * Add landmark roles to navigation (WCAG 4.1.2).
 *
 * @param string $nav_menu Nav markup.
 * @return string Modified nav markup.
 */
function rms_add_nav_role( $nav_menu ) {
	return preg_replace( '/<nav/', '<nav role="navigation"', $nav_menu );
}
add_filter( 'wp_nav_menu', 'rms_add_nav_role' );

/**
 * Normalize Navigation block ARIA to avoid menubar pattern for site navigation.
 *
 * - Remove role="menubar|menu|menuitem"
 * - Remove aria-label from the inner <ul>; prefer label on <nav>
 * - Ensure <nav> has role="navigation"
 *
 * @param string $block_content Rendered block HTML.
 * @param array  $block         Block data (name, attrs, etc.).
 * @return string Filtered HTML.
 */
function rms_normalize_core_navigation_aria( $block_content, $block ) {
	if ( empty( $block['blockName'] ) || 'core/navigation' !== $block['blockName'] ) {
		return $block_content;
	}

	// Remove ARIA menu roles not appropriate for site navigation.
	$block_content = preg_replace( '/\srole="(?:menubar|menu|menuitem)"/i', '', $block_content );

	// Remove aria-label from the inner UL (label should live on the NAV element).
	$block_content = preg_replace( '/(<ul[^>]*?)\saria-label="[^"]*"([^>]*>)/i', '$1$2', $block_content );

	// Ensure nav has role="navigation" if not already present.
	if ( ! preg_match( '/<nav[^>]*\srole="navigation"/i', $block_content ) ) {
		$block_content = preg_replace( '/<nav(\s|>)/i', '<nav role="navigation"$1', $block_content, 1 );
	}

	return $block_content;
}
add_filter( 'render_block', 'rms_normalize_core_navigation_aria', 20, 2 );

/**
 * Resolve the localized Verify Certificate URL.
 *
 * @return string URL for the current locale, or empty string if unresolved.
 */
function rms_resolve_verify_url() {
	$url          = '';
	$verify_pages = get_option( 'rms_verify_pages', array() );
	$current_lang = function_exists( 'pll_current_language' ) ? pll_current_language() : null;

	if ( $current_lang && isset( $verify_pages[ $current_lang ] ) ) {
		$page_id   = (int) $verify_pages[ $current_lang ];
		$permalink = get_permalink( $page_id );
		if ( $permalink ) {
			$url = $permalink;
		}
	}

	if ( empty( $url ) ) {
		$base = get_page_by_path( 'verify-certificate', OBJECT, 'page' );
		if ( $base && ! is_wp_error( $base ) ) {
			$target_id = $base->ID;
			if ( $current_lang && function_exists( 'pll_get_post' ) ) {
				$translated = pll_get_post( $base->ID, $current_lang );
				if ( $translated ) {
					$target_id = $translated;
				}
			}
			$maybe = get_permalink( $target_id );
			if ( $maybe ) {
				$url = $maybe;
			}
		}
	}

	if ( empty( $url ) && class_exists( 'RMS_Multilingual_Content' ) ) {
		$url = RMS_Multilingual_Content::get_content( 'verify_url' );
	}

	return $url;
}

/**
 * Localize the placeholder navigation link with class "rms-verify-link".
 *
 * Converts it into a per-locale Verify Certificate link with the same look
 * and feel as other navigation items (uses core navigation markup).
 *
 * @param array $block Parsed block data prior to render.
 * @return array Possibly modified block data.
 */
function rms_localize_verify_nav_link( $block ) {
	if ( empty( $block['blockName'] ) || 'core/navigation-link' !== $block['blockName'] ) {
		return $block;
	}

	$attrs     = isset( $block['attrs'] ) ? $block['attrs'] : array();
	$class     = isset( $attrs['className'] ) ? (string) $attrs['className'] : '';
	$has_class = ( false !== strpos( ' ' . $class . ' ', ' rms-verify-link ' ) );
	if ( ! $has_class ) {
		return $block;
	}

	// Compute localized label and URL.
	$label = '';
	if ( class_exists( 'RMS_Multilingual_Content' ) ) {
		$label = RMS_Multilingual_Content::get_content( 'verify_label' );
	}
	if ( empty( $label ) ) {
		$label = __( 'Verify Certificate', 'rms' );
	}

	$url = rms_resolve_verify_url();

	// Apply to block attributes so core will render the proper nav item.
	$block['attrs']['label'] = $label;
	if ( ! empty( $url ) ) {
		$block['attrs']['url'] = $url;
	}

	// Ensure it does not open in a new tab.
	$block['attrs']['opensInNewTab'] = false;

	return $block;
}
add_filter( 'render_block_data', 'rms_localize_verify_nav_link', 10 );

/**
 * Inject a Verify navigation item into the header's primary navigation when
 * the menu is DB-driven and our placeholder block isn't present.
 *
 * We scope by checking for the Header-nav class on the rendered nav markup.
 *
 * @param string $block_content Rendered HTML.
 * @param array  $block         Block data.
 * @return string Modified HTML.
 */
function rms_inject_verify_link_in_primary_navigation( $block_content, $block ) {
	if ( empty( $block['blockName'] ) || 'core/navigation' !== $block['blockName'] ) {
		return $block_content;
	}

	// Only affect the header primary nav (has class Header-nav in our theme).
	if ( false === strpos( $block_content, 'Header-nav' ) ) {
		return $block_content;
	}

	$url   = rms_resolve_verify_url();
	$label = class_exists( 'RMS_Multilingual_Content' ) ? RMS_Multilingual_Content::get_content( 'verify_label' ) : __( 'Verify Certificate', 'rms' );
	if ( empty( $label ) ) {
		$label = __( 'Verify Certificate', 'rms' );
	}

	// Avoid duplicates if a nav item already points to the verify URL or label.
	if ( ( $url && false !== strpos( $block_content, esc_url( $url ) ) ) || false !== stripos( $block_content, $label ) ) {
		return $block_content;
	}

	// Render a core/navigation-link block so markup matches theme/editor output.
	$verify_li = render_block(
		array(
			'blockName' => 'core/navigation-link',
			'attrs'     => array(
				'label'          => $label,
				'url'            => $url ?: '#',
				'opensInNewTab'  => false,
				'kind'           => 'custom',
				'className'      => 'rms-verify-link',
			),
			'innerBlocks' => array(),
		)
	);

	if ( empty( $verify_li ) ) {
		return $block_content;
	}

	// Insert as the second item (after the first <li> in the nav container).
	$pattern = '/(<ul[^>]*class="[^"]*wp-block-navigation__container[^"]*"[^>]*>)(\s*<li[\s\S]*?<\/li>)([\s\S]*?<\/ul>)/i';
	if ( preg_match( $pattern, $block_content ) ) {
		$block_content = preg_replace( $pattern, '$1$2' . $verify_li . '$3', $block_content, 1 );
	} else {
		// Fallback: append before closing </ul> of any navigation container.
		$block_content = preg_replace( '/(<ul[^>]*wp-block-navigation__container[^>]*>)([\s\S]*?)(<\/ul>)/i', '$1$2' . $verify_li . '$3', $block_content, 1 );
	}

	return $block_content;
}
add_filter( 'render_block', 'rms_inject_verify_link_in_primary_navigation', 30, 2 );

/**
 * Resolve the localized Blog URL.
 *
 * @return string URL for the current locale, or empty string if unresolved.
 */
function rms_resolve_blog_url() {
	if ( class_exists( 'RMS_Multilingual_Content' ) ) {
		$url = RMS_Multilingual_Content::get_blog_url();
		if ( $url ) {
			return $url;
		}
	}
	return '';
}

/**
 * Localize the placeholder navigation link with class "rms-blog-link".
 *
 * Converts it into a per-locale Blog link with the same look and feel as
 * other navigation items (uses core navigation markup).
 *
 * @param array $block Parsed block data prior to render.
 * @return array Possibly modified block data.
 */
function rms_localize_blog_nav_link( $block ) {
	if ( empty( $block['blockName'] ) || 'core/navigation-link' !== $block['blockName'] ) {
		return $block;
	}

	$attrs     = isset( $block['attrs'] ) ? $block['attrs'] : array();
	$class     = isset( $attrs['className'] ) ? (string) $attrs['className'] : '';
	$has_class = ( false !== strpos( ' ' . $class . ' ', ' rms-blog-link ' ) );
	if ( ! $has_class ) {
		return $block;
	}

	$label = '';
	if ( class_exists( 'RMS_Multilingual_Content' ) ) {
		$label = RMS_Multilingual_Content::get_content( 'blog_label' );
	}
	if ( empty( $label ) ) {
		$label = __( 'Blog', 'rms' );
	}

	$url = rms_resolve_blog_url();

	$block['attrs']['label'] = $label;
	if ( ! empty( $url ) ) {
		$block['attrs']['url'] = $url;
	}
	$block['attrs']['opensInNewTab'] = false;

	return $block;
}
add_filter( 'render_block_data', 'rms_localize_blog_nav_link', 10 );

/**
 * Inject a Blog navigation item into the header's primary navigation when
 * the menu is DB-driven and our placeholder block isn't present.
 *
 * Inserts before the Verify link by using a slightly earlier priority.
 *
 * @param string $block_content Rendered HTML.
 * @param array  $block         Block data.
 * @return string Modified HTML.
 */
function rms_inject_blog_link_in_primary_navigation( $block_content, $block ) {
	if ( empty( $block['blockName'] ) || 'core/navigation' !== $block['blockName'] ) {
		return $block_content;
	}

	// Only affect the header primary nav (has class Header-nav in our theme).
	if ( false === strpos( $block_content, 'Header-nav' ) ) {
		return $block_content;
	}

	$url   = rms_resolve_blog_url();
	$label = class_exists( 'RMS_Multilingual_Content' ) ? RMS_Multilingual_Content::get_content( 'blog_label' ) : __( 'Blog', 'rms' );
	if ( empty( $label ) ) {
		$label = __( 'Blog', 'rms' );
	}

	// Avoid duplicates if a nav item already points to the blog URL or label.
	if ( ( $url && false !== strpos( $block_content, esc_url( $url ) ) ) || false !== stripos( $block_content, $label ) ) {
		return $block_content;
	}

	$blog_li = render_block(
		array(
			'blockName'   => 'core/navigation-link',
			'attrs'       => array(
				'label'         => $label,
				'url'           => $url ?: '#',
				'opensInNewTab' => false,
				'kind'          => 'custom',
				'className'     => 'rms-blog-link',
			),
			'innerBlocks' => array(),
		)
	);

	if ( empty( $blog_li ) ) {
		return $block_content;
	}

	// Insert as the second item (after the first <li> in the nav container).
	$pattern = '/(<ul[^>]*class="[^"]*wp-block-navigation__container[^"]*"[^>]*>)(\s*<li[\s\S]*?<\/li>)([\s\S]*?<\/ul>)/i';
	if ( preg_match( $pattern, $block_content ) ) {
		$block_content = preg_replace( $pattern, '$1$2' . $blog_li . '$3', $block_content, 1 );
	} else {
		// Fallback: append before closing </ul> of any navigation container.
		$block_content = preg_replace( '/(<ul[^>]*wp-block-navigation__container[^>]*>)([\s\S]*?)(<\/ul>)/i', '$1$2' . $blog_li . '$3', $block_content, 1 );
	}

	return $block_content;
}
add_filter( 'render_block', 'rms_inject_blog_link_in_primary_navigation', 29, 2 );

/**
 * Enqueue accessibility helper script (WCAG 2.4.1).
 *
 * @return void
 */
function rms_enqueue_accessibility_scripts() {
	wp_enqueue_script(
		'rms-accessibility',
		get_template_directory_uri() . '/assets/js/accessibility.js',
		array(),
		wp_get_theme()->get( 'Version' ),
		true
	);
}
add_action( 'wp_enqueue_scripts', 'rms_enqueue_accessibility_scripts' );

/**
 * Debug function to see which template is being used.
 * Remove this after debugging.
 *
 * @return void
 */
function rms_debug_template() {
	if ( current_user_can( 'manage_options' ) ) {
		global $template;
		echo '<!-- Current template: ' . esc_html( basename( $template ) ) . ' -->';
	}
}
add_action( 'wp_head', 'rms_debug_template' );

/**
 * Clear database templates admin action.
 * Access via: http://yoursite.local/?rms_clear_templates=1.
 *
 * @return void
 */
function rms_clear_database_templates() {
	// Security checks.
	if ( ! isset( $_GET['rms_clear_templates'] ) || '1' !== $_GET['rms_clear_templates'] ) {
		return;
	}

	// Require nonce to avoid processing without verification.
	if ( ! isset( $_GET['_wpnonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_GET['_wpnonce'] ) ), 'rms_clear_templates' ) ) {
		return;
	}

	if ( ! current_user_can( 'manage_options' ) ) {
		wp_die( 'Access denied. Administrator privileges required.' );
	}

	// Get all database-stored templates for this theme.
	$templates = get_posts(
		array(
			'post_type'   => 'wp_template',
			'post_status' => array( 'publish', 'draft', 'trash' ),
			'numberposts' => -1,
			'meta_query'  => array(
				array(
					'key'     => '_wp_theme',
					'value'   => get_stylesheet(),
					'compare' => '=',
				),
			),
		)
	);

	$deleted_count  = 0;
	$template_names = array();

	foreach ( $templates as $template ) {
		$template_names[] = $template->post_name;
		if ( wp_delete_post( $template->ID, true ) ) {
			++$deleted_count;
		}
	}

	// Show results.
	$names_str = implode( ', ', $template_names );
	if ( '' === $names_str ) {
		$names_str = 'none found';
	}
	$message = sprintf(
		'Database Template Cleanup Complete!\n\nDeleted %d templates: %s\n\nYour file-based templates should now work properly.\n\nClick OK to return home.',
		$deleted_count,
		$names_str
	);

	echo '<script>alert(' . wp_json_encode( $message ) . '); window.location.href = ' . wp_json_encode( home_url() ) . ';</script>';
	exit;
}
add_action( 'init', 'rms_clear_database_templates' );

// Include only required theme files for block theme.
$security_file = __DIR__ . '/inc/class-rms-security.php';
if ( file_exists( $security_file ) ) {
	require_once $security_file;
}

$hero_section_file = __DIR__ . '/patterns/hero-section.php';
if ( file_exists( $hero_section_file ) ) {
	require_once $hero_section_file;
}

// Initialize security features (only if class exists).
if ( class_exists( 'RMS_Security' ) ) {
	RMS_Security::init();
}

/**
 * Simple language switcher shortcode for direct use.
 *
 * @return string HTML output.
 */
function rms_simple_language_switcher() {
	// Check if Polylang is active.
	if ( ! function_exists( 'pll_the_languages' ) ) {
		return '<div class="polylang-not-active">' . esc_html__( 'Polylang plugin is not active.', 'rms' ) . '</div>';
	}

	// Get current language.
	$current_lang = pll_current_language();
	$languages    = pll_the_languages(
		array(
			'show_flags'    => 1,
			'show_names'    => 1,
			'hide_if_empty' => 1,
			'force_home'    => 0,
			'hide_current'  => 0,
			'raw'           => 1,
		)
	);

	if ( empty( $languages ) ) {
		return '';
	}

	$output  = '<div class="simple-language-switcher">';
	$output .= '<ul class="language-list">';

	foreach ( $languages as $language ) {
		$class = $language['current_lang'] ? 'current-lang' : '';
		$flag  = isset( $language['flag'] ) ? $language['flag'] : '';

		$output .= '<li class="' . esc_attr( $class ) . '">';
		$output .= '<a href="' . esc_url( $language['url'] ) . '">';

		if ( $flag ) {
			$output .= '<span class="flag">' . esc_html( $flag ) . '</span>';
		}

		$output .= '<span class="lang-name">' . esc_html( $language['name'] ) . '</span>';
		$output .= '</a>';
		$output .= '</li>';
	}

	$output .= '</ul>';
	$output .= '</div>';

	return $output;
}
add_shortcode( 'rms_language_switcher', 'rms_simple_language_switcher' );

/**
 * Render callback for Verify Certificate Button block.
 *
 * @param array  $attributes Block attributes.
 * @param string $content    Inner content (unused).
 * @return string            HTML output.
 */
function rms_render_verify_button_block( $attributes, $content ) {
	$label      = trim( $attributes['label'] ?? '' );
	$url        = trim( $attributes['url'] ?? '' );
	$use_primary = ! empty( $attributes['usePrimary'] );

	// Fallback to multilingual content.
	if ( empty( $label ) && class_exists( 'RMS_Multilingual_Content' ) ) {
		$label = RMS_Multilingual_Content::get_content( 'verify_label' );
	}

	// Resolve URL by stored per-language verify page IDs for accurate permalinks.
	if ( empty( $url ) ) {
		$verify_pages = get_option( 'rms_verify_pages', array() );
		$current_lang = function_exists( 'pll_current_language' ) ? pll_current_language() : null;
		if ( $current_lang && isset( $verify_pages[ $current_lang ] ) ) {
			$page_id = (int) $verify_pages[ $current_lang ];
			$permalink = get_permalink( $page_id );
			if ( $permalink ) {
				$url = $permalink;
			}
		}

		// If still empty, try resolving via Polylang translation of the base English page.
		if ( empty( $url ) ) {
			$base = get_page_by_path( 'verify-certificate', OBJECT, 'page' );
			if ( $base && ! is_wp_error( $base ) ) {
				$target_id = $base->ID;
				if ( $current_lang && function_exists( 'pll_get_post' ) ) {
					$translated = pll_get_post( $base->ID, $current_lang );
					if ( $translated ) {
						$target_id = $translated;
					}
				}
				$maybe = get_permalink( $target_id );
				if ( $maybe ) {
					$url = $maybe;
				}
			}
		}
		// Fallback to content defaults if still empty.
		if ( empty( $url ) && class_exists( 'RMS_Multilingual_Content' ) ) {
			$url = RMS_Multilingual_Content::get_content( 'verify_url' );
		}
	}

	$classes = 'wp-block-button__link wp-element-button';
	if ( $use_primary ) {
		$classes .= ' has-background-color has-secondary-background-color has-text-color has-background';
	}

	$wrapper_attrs = get_block_wrapper_attributes( array( 'class' => 'wp-block-button rms-verify-button' ) );

	ob_start();
	?>
	<div <?php echo $wrapper_attrs; // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped ?>>
		<a class="<?php echo esc_attr( $classes ); ?>" href="<?php echo esc_url( $url ); ?>">
			<?php echo esc_html( $label ?: __( 'Verify Certificate', 'rms' ) ); ?>
		</a>
	</div>
	<?php
	return ob_get_clean();
}

/**
 * Render callback for Counter block.
 *
 * @param array  $attributes Block attributes.
 * @param string $content    Inner content (unused).
 * @return string            HTML output.
 */
function rms_render_counter_block( $attributes, $content ) {
	$value         = isset( $attributes['value'] ) ? floatval( $attributes['value'] ) : 0;
	$prefix        = (string) ( $attributes['prefix'] ?? '' );
	$suffix        = (string) ( $attributes['suffix'] ?? '' );
	$label         = (string) ( $attributes['label'] ?? '' );
	$label_key     = (string) ( $attributes['labelKey'] ?? '' );
	$duration      = isset( $attributes['duration'] ) ? intval( $attributes['duration'] ) : 1200;
	$grouping      = ! empty( $attributes['grouping'] );
	$reduce_motion = ! empty( $attributes['reduceMotion'] );

	if ( empty( $label ) && ! empty( $label_key ) && class_exists( 'RMS_Multilingual_Content' ) ) {
		$label = RMS_Multilingual_Content::get_content( $label_key );
	}

	// Translate label via theme text domain when present so previously-inserted
	// English labels (e.g., DB content) localize at render time.
	if ( $label ) {
		$maybe_translated = __( $label, 'rms' );

		// If translation is missing and we're not in English, try mapping from
		// English defaults to current language defaults to avoid DB edits.
		if (
			$maybe_translated === $label
			&& function_exists( 'pll_current_language' )
			&& class_exists( 'RMS_Multilingual_Content' )
		) {
			$current_lang = pll_current_language();
			if ( $current_lang && 'en' !== $current_lang ) {
				$en_items   = RMS_Multilingual_Content::get_content( 'stats_items', 'en' );
				$loc_items  = RMS_Multilingual_Content::get_content( 'stats_items', $current_lang );
				if ( is_array( $en_items ) && is_array( $loc_items ) ) {
					foreach ( $en_items as $idx => $en_item ) {
						if ( isset( $en_item['label'] ) && (string) $en_item['label'] === $label && isset( $loc_items[ $idx ]['label'] ) ) {
							$maybe_translated = (string) $loc_items[ $idx ]['label'];
							break;
						}
					}
				}
			}
		}

		$label = $maybe_translated;
	}

	$wrapper_attrs = get_block_wrapper_attributes( array( 'class' => 'rms-counter' ) );

	// Human-friendly final text for screen readers.
	$final_text = trim( $prefix . ' ' . number_format_i18n( $value ) . ' ' . $suffix );

	ob_start();
	?>
	<div <?php echo $wrapper_attrs; // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped ?>
		data-value="<?php echo esc_attr( $value ); ?>"
		data-duration="<?php echo esc_attr( max( 0, $duration ) ); ?>"
		data-prefix="<?php echo esc_attr( $prefix ); ?>"
		data-suffix="<?php echo esc_attr( $suffix ); ?>"
		data-grouping="<?php echo esc_attr( $grouping ? '1' : '0' ); ?>"
		data-reduce-motion="<?php echo esc_attr( $reduce_motion ? '1' : '0' ); ?>"
	>
		<div class="rms-counter__number" aria-hidden="true">
			<span class="rms-counter__value">0</span>
		</div>
		<span class="screen-reader-text rms-counter__sr"><?php echo esc_html( $final_text ); ?></span>
		<?php if ( $label ) : ?>
			<div class="rms-counter__label"><?php echo esc_html( $label ); ?></div>
		<?php endif; ?>
	</div>
	<?php
	return ob_get_clean();
}

/**
 * Filter core/heading blocks to localize the Stats section title if it was
 * saved in English in the database. This targets the common heading text and
 * replaces it with the per-locale default at render time.
 *
 * @param string $block_content Rendered block HTML.
 * @param array  $block         Block data.
 * @return string               Potentially localized block HTML.
 */
function rms_filter_stats_heading_title( $block_content, $block ) {
	if ( empty( $block['blockName'] ) || 'core/heading' !== $block['blockName'] ) {
		return $block_content;
	}

	$plain = trim( wp_strip_all_tags( $block_content ) );
	if ( 'Our Impact in Numbers' !== $plain ) {
		return $block_content;
	}

	$current_lang = function_exists( 'pll_current_language' ) ? pll_current_language() : 'en';
	// Only swap when not English to avoid unnecessary work.
	if ( 'en' === $current_lang ) {
		return $block_content;
	}

	$title = class_exists( 'RMS_Multilingual_Content' )
		? RMS_Multilingual_Content::get_content( 'stats_title', $current_lang )
		: __( 'Our Impact in Numbers', 'rms' );

	if ( ! $title ) {
		return $block_content;
	}

	// Replace the inner text content of the first heading tag.
	$localized = preg_replace( '/(<h[1-6][^>]*>)(.*?)(<\/h[1-6]>)/is', '$1' . esc_html( $title ) . '$3', $block_content, 1 );
	return $localized ? $localized : $block_content;
}
add_filter( 'render_block', 'rms_filter_stats_heading_title', 10, 2 );
