<?php
/**
 * Multilingual Content Management for RMS Theme
 *
 * @package RMS
 * @since 1.0.0
 */

// Prevent direct access.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Class for managing multilingual content.
 */
class RMS_Multilingual_Content {

	/**
	 * Default content for different languages.
	 *
	 * @var array
	 */
	private static $default_content = array(
		'en' => array(
			'site_title'        => 'Rimess Almasry',
			'site_description'  => 'Excellence in Multiple Fields',
			'hero_title'        => 'Welcome to Excellence',
			'hero_subtitle'     => 'Discover our premium services and innovative solutions',
			'hero_cta'          => 'Get Started',
			// Header CTA banner (not rendered anymore, kept for completeness/localization references).
			'header_cta_title'    => 'Apply Now',
			'header_cta_subtitle' => "(Professional Master's & PhD)",
			'header_cta_button'   => 'Register here',
			'header_cta_url'      => '/application-form/',
			'verify_label'      => 'Verify Certificate',
			'verify_url'        => '/verify-certificate/',
			'services_title'    => 'Our Services',
			'services_subtitle' => 'Professional solutions tailored to your needs',
			'about_title'       => 'About Us',
			'about_content'     => 'We are dedicated to providing exceptional services across multiple industries. Our commitment to excellence drives everything we do.',
			'blog_label'        => 'Blog',
			'contact_title'     => 'Get in Touch',
			'contact_subtitle'  => 'Ready to start your journey with us?',
			'footer_text'       => 'Excellence in Multiple Fields',
			'copyright'         => '© 2025 Rimess Almasry. All rights reserved.',
			// Stats (counters)
			'stats_title'       => 'Our Impact in Numbers',
			'stats_items'       => array(
				array( 'label' => 'Trainees', 'value' => 12000, 'suffix' => '+' ),
				array( 'label' => 'Trainers', 'value' => 350, 'suffix' => '+' ),
				array( 'label' => 'International Partners', 'value' => 25, 'suffix' => '+' ),
				array( 'label' => 'Excellence Achieved', 'value' => 98, 'suffix' => '%' ),
			),
		),
		'fr' => array(
			'site_title'        => 'Rimess Almasry',
			'site_description'  => "L'Excellence dans Plusieurs Domaines",
			'hero_title'        => "Bienvenue à l'Excellence",
			'hero_subtitle'     => 'Découvrez nos services premium et solutions innovantes',
			'hero_cta'          => 'Commencer',
			// Header CTA banner (not rendered anymore, kept for completeness/localization references).
			'header_cta_title'    => 'Postulez maintenant',
			'header_cta_subtitle' => '(Master et Doctorat Professionnels)',
			'header_cta_button'   => 'Inscrivez-vous ici',
			'header_cta_url'      => '/fr/application-form/',
			'verify_label'      => 'Vérifier le certificat',
			'verify_url'        => '/fr/verify-certificate-fr/',
			'services_title'    => 'Nos Services',
			'services_subtitle' => 'Solutions professionnelles adaptées à vos besoins',
			'about_title'       => 'À Propos',
			'about_content'     => "Nous nous consacrons à fournir des services exceptionnels dans plusieurs industries. Notre engagement envers l'excellence guide tout ce que nous faisons.",
			'blog_label'        => 'Blog',
			'contact_title'     => 'Contactez-nous',
			'contact_subtitle'  => 'Prêt à commencer votre parcours avec nous?',
			'footer_text'       => "L'Excellence dans Plusieurs Domaines",
			'copyright'         => '© 2025 Rimess Almasry. Tous droits réservés.',
			// Statistiques (compteurs)
			'stats_title'       => 'Notre impact en chiffres',
			'stats_items'       => array(
				array( 'label' => 'Stagiaires', 'value' => 12000, 'suffix' => '+' ),
				array( 'label' => 'Formateurs', 'value' => 350, 'suffix' => '+' ),
				array( 'label' => 'Partenaires Internationaux', 'value' => 25, 'suffix' => '+' ),
				array( 'label' => 'Excellence Atteinte', 'value' => 98, 'suffix' => '%' ),
			),
		),
		'ar' => array(
			'site_title'        => 'ريماس المصري',
			'site_description'  => 'التميز في مجالات متعددة',
			'hero_title'        => 'مرحباً بكم في التميز',
			'hero_subtitle'     => 'اكتشف خدماتنا المتميزة وحلولنا المبتكرة',
			'hero_cta'          => 'ابدأ الآن',
			// Header CTA banner (not rendered anymore, kept for completeness/localization references).
			'header_cta_title'    => 'تقدم الآن',
			'header_cta_subtitle' => '(الماجستير والدكتوراه المهنية )',
			'header_cta_button'   => 'التسجيل هنا',
			'header_cta_url'      => '/ar/application-form/',
			'verify_label'      => 'تحقق من الشهادة',
			'verify_url'        => '/ar/verify-certificate-ar/',
			'services_title'    => 'خدماتنا',
			'services_subtitle' => 'حلول مهنية مصممة خصيصاً لاحتياجاتك',
			'about_title'       => 'من نحن',
			'about_content'     => 'نحن ملتزمون بتقديم خدمات استثنائية عبر صناعات متعددة. التزامنا بالتميز يوجه كل ما نقوم به.',
			'blog_label'        => 'المدونة',
			'contact_title'     => 'تواصل معنا',
			'contact_subtitle'  => 'مستعد لبدء رحلتك معنا؟',
			'footer_text'       => 'التميز في مجالات متعددة',
			'copyright'         => '© 2025 ريماس المصري. جميع الحقوق محفوظة.',
			// الإحصائيات (العدادات)
			'stats_title'       => 'أثرنا بالأرقام',
			'stats_items'       => array(
				array( 'label' => 'المتدربين', 'value' => 12000, 'suffix' => '+' ),
				array( 'label' => 'المدربين', 'value' => 350, 'suffix' => '+' ),
				array( 'label' => 'شركاء دوليون', 'value' => 25, 'suffix' => '+' ),
				array( 'label' => 'تحقيق التميز', 'value' => 98, 'suffix' => '%' ),
			),
		),
	);

	/**
	 * Service cards content for different languages.
	 */
	private static $service_cards = array(
		'en' => array(
			array(
				'icon'        => '🚀',
				'title'       => 'Innovation',
				'description' => 'Cutting-edge solutions powered by the latest technology and creative thinking.',
				'link'        => '/services/innovation/',
			),
			array(
				'icon'        => '💼',
				'title'       => 'Business Solutions',
				'description' => 'Comprehensive business strategies to help your company grow and succeed.',
				'link'        => '/services/business/',
			),
			array(
				'icon'        => '🎯',
				'title'       => 'Consulting',
				'description' => 'Expert advice and guidance to navigate complex challenges and opportunities.',
				'link'        => '/services/consulting/',
			),
			array(
				'icon'        => '🔧',
				'title'       => 'Technical Support',
				'description' => 'Professional technical assistance and maintenance for all your systems.',
				'link'        => '/services/support/',
			),
		),
		'fr' => array(
			array(
				'icon'        => '🚀',
				'title'       => 'Innovation',
				'description' => 'Solutions de pointe alimentées par la dernière technologie et la pensée créative.',
				'link'        => '/fr/services/innovation/',
			),
			array(
				'icon'        => '💼',
				'title'       => 'Solutions Business',
				'description' => 'Stratégies commerciales complètes pour aider votre entreprise à croître et réussir.',
				'link'        => '/fr/services/business/',
			),
			array(
				'icon'        => '🎯',
				'title'       => 'Conseil',
				'description' => 'Conseils d\'experts et orientation pour naviguer dans les défis et opportunités complexes.',
				'link'        => '/fr/services/consulting/',
			),
			array(
				'icon'        => '🔧',
				'title'       => 'Support Technique',
				'description' => 'Assistance technique professionnelle et maintenance pour tous vos systèmes.',
				'link'        => '/fr/services/support/',
			),
		),
		'ar' => array(
			array(
				'icon'        => '🚀',
				'title'       => 'الابتكار',
				'description' => 'حلول متطورة مدعومة بأحدث التقنيات والتفكير الإبداعي.',
				'link'        => '/ar/services/innovation/',
			),
			array(
				'icon'        => '💼',
				'title'       => 'الحلول التجارية',
				'description' => 'استراتيجيات تجارية شاملة لمساعدة شركتك على النمو والنجاح.',
				'link'        => '/ar/services/business/',
			),
			array(
				'icon'        => '🎯',
				'title'       => 'الاستشارات',
				'description' => 'نصائح وإرشادات الخبراء للتنقل في التحديات والفرص المعقدة.',
				'link'        => '/ar/services/consulting/',
			),
			array(
				'icon'        => '🔧',
				'title'       => 'الدعم التقني',
				'description' => 'مساعدة تقنية مهنية وصيانة لجميع أنظمتك.',
				'link'        => '/ar/services/support/',
			),
		),
	);

	/**
	 * Initialize the multilingual content system.
	 */
	public static function init() {
		add_action( 'init', array( __CLASS__, 'setup_translated_pages' ) );
		add_action( 'init', array( __CLASS__, 'ensure_verify_pages' ), 20 );
		add_action( 'init', array( __CLASS__, 'ensure_blog_pages' ), 25 );
		add_action( 'wp', array( __CLASS__, 'setup_front_page_content' ) );
		add_filter( 'the_content', array( __CLASS__, 'filter_front_page_content' ) );
		add_action( 'wp_head', array( __CLASS__, 'add_language_meta_tags' ) );
		add_action( 'pre_get_posts', array( __CLASS__, 'route_localized_blog_pages' ) );
	}

	/**
	 * Get content for current language.
	 *
	 * @param string      $key  Content key.
	 * @param string|null $lang Language code.
	 * @return string Content value.
	 */
	public static function get_content( $key, $lang = null ) {
		if ( ! $lang ) {
			$lang = function_exists( 'pll_current_language' ) ? pll_current_language() : 'en';
		}

		// Fallback to English if language not found.
		if ( ! isset( self::$default_content[ $lang ] ) ) {
			$lang = 'en';
		}

		return self::$default_content[ $lang ][ $key ] ?? '';
	}

	/**
	 * Get service cards for current language.
	 *
	 * @param string|null $lang Language code.
	 * @return array Cards.
	 */
	public static function get_service_cards( $lang = null ) {
		if ( ! $lang ) {
			$lang = function_exists( 'pll_current_language' ) ? pll_current_language() : 'en';
		}

		// Fallback to English if language not found.
		if ( ! isset( self::$service_cards[ $lang ] ) ) {
			$lang = 'en';
		}

		return self::$service_cards[ $lang ] ?? array();
	}

	/**
	 * Setup translated pages automatically.
	 */
	public static function setup_translated_pages() {
		if ( ! function_exists( 'pll_languages_list' ) ) {
			return;
		}

		// Only run this once.
		if ( get_option( 'rms_multilingual_pages_created' ) ) {
			return;
		}

		$languages      = pll_languages_list();
		$front_page_ids = array();

		foreach ( $languages as $lang ) {
			$page_id = self::create_translated_front_page( $lang );
			if ( $page_id ) {
				$front_page_ids[ $lang ] = $page_id;
			}
		}

		// Set front pages for each language.
		if ( ! empty( $front_page_ids ) ) {
			foreach ( $front_page_ids as $lang => $page_id ) {
				if ( function_exists( 'pll_set_post_language' ) ) {
					pll_set_post_language( $page_id, $lang );
				}
			}

			// Link the translations.
			if ( function_exists( 'pll_save_post_translations' ) && 1 < count( $front_page_ids ) ) {
				pll_save_post_translations( $front_page_ids );
			}
		}

		update_option( 'rms_multilingual_pages_created', true );
	}

	/**
	 * Create a translated front page.
	 *
	 * @param string $lang Language code.
	 * @return int|false Page ID or false.
	 */
	private static function create_translated_front_page( $lang ) {
		$content = self::$default_content[ $lang ] ?? self::$default_content['en'];

		$page_content = self::generate_front_page_content( $lang );

		$page_data = array(
			'post_title'   => $content['site_title'],
			'post_content' => $page_content,
			'post_status'  => 'publish',
			'post_type'    => 'page',
			'post_name'    => 'en' === $lang ? 'home' : "home-{$lang}",
			'meta_input'   => array(
				'_wp_page_template' => 'index.php',
			),
		);

		$page_id = wp_insert_post( $page_data );

		if ( $page_id && ! is_wp_error( $page_id ) ) {
			// Set as front page for this language if it's the default language.
			if ( function_exists( 'pll_default_language' ) && pll_default_language() === $lang ) {
				update_option( 'page_on_front', $page_id );
				update_option( 'show_on_front', 'page' );
			}

			return $page_id;
		}

		return false;
	}

	/**
	 * Ensure per-language Verify Certificate pages exist and are translated.
	 */
	public static function ensure_verify_pages() {
		if ( ! function_exists( 'pll_languages_list' ) ) {
			return;
		}

		$languages = pll_languages_list();
		$created   = array();

		foreach ( $languages as $lang ) {
			$content     = self::$default_content[ $lang ] ?? self::$default_content['en'];
			// Use a language-specific slug to avoid conflicts, but we will resolve URLs dynamically.
			$verify_slug = ( 'en' === $lang ) ? 'verify-certificate' : "verify-certificate-{$lang}";
			$title_map   = array(
				'en' => __( 'Verify Certificate', 'rms' ),
				'fr' => __( 'Vérifier le certificat', 'rms' ),
				'ar' => __( 'تحقق من الشهادة', 'rms' ),
			);
			$title       = $title_map[ $lang ] ?? $content['verify_label'];

			$existing = get_page_by_path( $verify_slug, OBJECT, 'page' );
			if ( ! $existing ) {
				// Do not pass a placeholder attribute; let the plugin localize it.
				$page_id     = wp_insert_post(
					array(
						'post_title'   => $title,
						'post_name'    => $verify_slug,
						'post_status'  => 'publish',
						'post_type'    => 'page',
						'post_content' => '[rms_verify_certificate]',
					)
				);
				if ( $page_id && ! is_wp_error( $page_id ) ) {
					$created[ $lang ] = (int) $page_id;
					if ( function_exists( 'pll_set_post_language' ) ) {
						pll_set_post_language( $page_id, $lang );
					}
				}
			} else {
				$created[ $lang ] = (int) $existing->ID;
				// Ensure language is set for existing page so translations can be linked.
				if ( function_exists( 'pll_set_post_language' ) ) {
					pll_set_post_language( (int) $existing->ID, $lang );
				}
				// Retrofit: remove hardcoded placeholder from shortcode if present to allow plugin localization.
				$content = get_post_field( 'post_content', $existing->ID );
				if ( false !== strpos( $content, '[rms_verify_certificate' ) && false !== strpos( $content, 'placeholder=' ) ) {
					$updated = preg_replace( '/\[rms_verify_certificate\s+placeholder="[^"]*"\s*\]/', '[rms_verify_certificate]', $content );
					if ( is_string( $updated ) && $updated !== $content ) {
						wp_update_post( array( 'ID' => (int) $existing->ID, 'post_content' => $updated ) );
					}
				}
			}
		}

		// Link verify pages across languages and store mapping for lookups.
		if ( function_exists( 'pll_save_post_translations' ) && ! empty( $created ) ) {
			pll_save_post_translations( $created );
		}

		// Persist mapping to an option so other components can resolve permalinks reliably.
		if ( ! empty( $created ) ) {
			update_option( 'rms_verify_pages', array_map( 'intval', $created ) );
		}
	}

	/**
	 * Ensure per-language Blog pages exist and are translated, and set posts page.
	 */
	public static function ensure_blog_pages() {
		if ( ! function_exists( 'pll_languages_list' ) ) {
			return;
		}

		$languages = pll_languages_list();
		$created   = array();

		$labels = array(
			'en' => __( 'Blog', 'rms' ),
			'fr' => __( 'Blog', 'rms' ),
			'ar' => __( 'المدونة', 'rms' ),
		);

		foreach ( $languages as $lang ) {
			$slug    = ( 'en' === $lang ) ? 'blog' : "blog-{$lang}";
			$title   = $labels[ $lang ] ?? self::$default_content[ $lang ]['blog_label'] ?? 'Blog';
			$existing = get_page_by_path( $slug, OBJECT, 'page' );
			if ( ! $existing ) {
				$page_id = wp_insert_post(
					array(
						'post_title'   => $title,
						'post_name'    => $slug,
						'post_status'  => 'publish',
						'post_type'    => 'page',
						'post_content' => '',
					)
				);
				if ( $page_id && ! is_wp_error( $page_id ) ) {
					$created[ $lang ] = (int) $page_id;
					if ( function_exists( 'pll_set_post_language' ) ) {
						pll_set_post_language( $page_id, $lang );
					}
				}
			} else {
				$created[ $lang ] = (int) $existing->ID;
				if ( function_exists( 'pll_set_post_language' ) ) {
					pll_set_post_language( (int) $existing->ID, $lang );
				}
			}
		}

		// Link blog pages across languages and store mapping.
		if ( function_exists( 'pll_save_post_translations' ) && ! empty( $created ) ) {
			pll_save_post_translations( $created );
		}

		if ( ! empty( $created ) ) {
			update_option( 'rms_blog_pages', array_map( 'intval', $created ) );
		}

		// Set default language blog page as the posts page.
		if ( function_exists( 'pll_default_language' ) ) {
			$default = pll_default_language();
			if ( $default && isset( $created[ $default ] ) ) {
				update_option( 'page_for_posts', (int) $created[ $default ] );
				update_option( 'show_on_front', 'page' );
			}
		}
	}

	/**
	 * Get the localized Blog page URL.
	 *
	 * @param string|null $lang Language code.
	 * @return string URL or empty.
	 */
	public static function get_blog_url( $lang = null ) {
		if ( ! $lang && function_exists( 'pll_current_language' ) ) {
			$lang = pll_current_language();
		}
		$map = get_option( 'rms_blog_pages', array() );
		if ( $lang && isset( $map[ $lang ] ) ) {
			$permalink = get_permalink( (int) $map[ $lang ] );
			if ( $permalink ) {
				return $permalink;
			}
		}
		return '';
	}

	/**
	 * Make localized Blog pages behave like the posts index in their language.
	 *
	 * This keeps the home page static while routing posts to /blog (and its translations).
	 *
	 * @param WP_Query $query Main query.
	 * @return void
	 */
	public static function route_localized_blog_pages( $query ) {
		if ( is_admin() || ! $query->is_main_query() ) {
			return;
		}

		$map = get_option( 'rms_blog_pages', array() );
		if ( empty( $map ) ) {
			return;
		}

		$current_lang = function_exists( 'pll_current_language' ) ? pll_current_language() : null;
		if ( ! $current_lang || ! isset( $map[ $current_lang ] ) ) {
			return;
		}

		$blog_id = (int) $map[ $current_lang ];

		// If that localized Blog page is requested, turn the query into the posts index.
		if ( $query->is_page( $blog_id ) ) {
			$query->set( 'page_id', '' );
			$query->set( 'pagename', '' );
			$query->set( 'post_type', 'post' );
			$query->set( 'posts_per_page', get_option( 'posts_per_page' ) );
			$query->is_page = false;
			$query->is_singular = false;
			$query->is_home = true;
		}
	}

	/**
	 * Generate front page content with blocks.
	 *
	 * @param string $lang Language code.
	 * @return string Block markup.
	 */
	private static function generate_front_page_content( $lang ) {
		$content       = self::$default_content[ $lang ] ?? self::$default_content['en'];
		$service_cards = self::get_service_cards( $lang );

		$blocks = array();

		// Hero Section.
		$blocks[] = '<!-- wp:group {"align":"full","style":{"spacing":{"padding":{"top":"4rem","bottom":"4rem"}}},"backgroundColor":"black","textColor":"white"} -->';
		$blocks[] = '<div class="wp-block-group alignfull has-white-color has-black-background-color has-text-color has-background" style="padding-top:4rem;padding-bottom:4rem">';
		$blocks[] = '<!-- wp:heading {"textAlign":"center","level":1,"fontSize":"huge"} -->';
		$blocks[] = '<h1 class="wp-block-heading has-text-align-center has-huge-font-size">' . esc_html( $content['hero_title'] ) . '</h1>';
		$blocks[] = '<!-- /wp:heading -->';
		$blocks[] = '<!-- wp:paragraph {"align":"center","fontSize":"large"} -->';
		$blocks[] = '<p class="has-text-align-center has-large-font-size">' . esc_html( $content['hero_subtitle'] ) . '</p>';
		$blocks[] = '<!-- /wp:paragraph -->';
		$blocks[] = '<!-- wp:buttons {"layout":{"type":"flex","justifyContent":"center"}} -->';
		$blocks[] = '<div class="wp-block-buttons">';
		$blocks[] = '<!-- wp:button {"backgroundColor":"gold","textColor":"black"} -->';
		$blocks[] = '<div class="wp-block-button"><a class="wp-block-button__link has-black-color has-gold-background-color has-text-color has-background wp-element-button">' . esc_html( $content['hero_cta'] ) . '</a></div>';
		$blocks[] = '<!-- /wp:button -->';
		$blocks[] = '</div>';
		$blocks[] = '<!-- /wp:buttons -->';
		$blocks[] = '</div>';
		$blocks[] = '<!-- /wp:group -->';

		// Services Section.
		$blocks[] = '<!-- wp:group {"align":"full","style":{"spacing":{"padding":{"top":"4rem","bottom":"4rem"}}}} -->';
		$blocks[] = '<div class="wp-block-group alignfull" style="padding-top:4rem;padding-bottom:4rem">';
		$blocks[] = '<!-- wp:heading {"textAlign":"center","level":2} -->';
		$blocks[] = '<h2 class="wp-block-heading has-text-align-center">' . esc_html( $content['services_title'] ) . '</h2>';
		$blocks[] = '<!-- /wp:heading -->';
		$blocks[] = '<!-- wp:paragraph {"align":"center"} -->';
		$blocks[] = '<p class="has-text-align-center">' . esc_html( $content['services_subtitle'] ) . '</p>';
		$blocks[] = '<!-- /wp:paragraph -->';

		// Service Cards.
		$blocks[] = '<!-- wp:columns {"align":"wide"} -->';
		$blocks[] = '<div class="wp-block-columns alignwide">';

		foreach ( $service_cards as $card ) {
			$blocks[] = '<!-- wp:column -->';
			$blocks[] = '<div class="wp-block-column">';
			$blocks[] = '<!-- wp:rms/service-card {"icon":"' . esc_attr( $card['icon'] ) . '","title":"' . esc_attr( $card['title'] ) . '","description":"' . esc_attr( $card['description'] ) . '","linkUrl":"' . esc_url( $card['link'] ) . '"} /-->';
			$blocks[] = '</div>';
			$blocks[] = '<!-- /wp:column -->';
		}

		$blocks[] = '</div>';
		$blocks[] = '<!-- /wp:columns -->';
		$blocks[] = '</div>';
		$blocks[] = '<!-- /wp:group -->';

		// Stats/Counter Section (inline, localized, accessible).
		$stats_title = isset( $content['stats_title'] ) ? $content['stats_title'] : __( 'Our Impact in Numbers', 'rms' );
		$items       = isset( $content['stats_items'] ) ? $content['stats_items'] : array();
		if ( empty( $items ) || ! is_array( $items ) ) {
			$items = array(
				array( 'label' => __( 'Trainees', 'rms' ), 'value' => 12000, 'suffix' => '+' ),
				array( 'label' => __( 'Trainers', 'rms' ), 'value' => 350, 'suffix' => '+' ),
				array( 'label' => __( 'International Partners', 'rms' ), 'value' => 25, 'suffix' => '+' ),
				array( 'label' => __( 'Excellence Achieved', 'rms' ), 'value' => 98, 'suffix' => '%' ),
			);
		}

		$blocks[] = '<!-- wp:group {"align":"full","style":{"spacing":{"padding":{"top":"3rem","bottom":"3rem"}}},"backgroundColor":"primary-dark","className":"rms-stats"} -->';
		$blocks[] = '<div class="wp-block-group alignfull has-primary-dark-background-color has-background rms-stats" style="padding-top:3rem;padding-bottom:3rem">';
		$blocks[] = '<!-- wp:group {"align":"wide","layout":{"type":"constrained"}} -->';
		$blocks[] = '<div class="wp-block-group alignwide">';
		$blocks[] = '<!-- wp:heading {"textAlign":"center","level":2,"style":{"typography":{"fontWeight":"800"}}} -->';
		$blocks[] = '<h2 class="wp-block-heading has-text-align-center">' . esc_html( $stats_title ) . '</h2>';
		$blocks[] = '<!-- /wp:heading -->';
		$blocks[] = '<!-- wp:spacer {"height":"20px"} --><div style="height:20px" aria-hidden="true" class="wp-block-spacer"></div><!-- /wp:spacer -->';
		$blocks[] = '<!-- wp:columns {"align":"wide","style":{"spacing":{"blockGap":"2rem"}}} -->';
		$blocks[] = '<div class="wp-block-columns alignwide">';
		foreach ( $items as $item ) {
			$value  = isset( $item['value'] ) ? (int) $item['value'] : 0;
			$label  = isset( $item['label'] ) ? (string) $item['label'] : '';
			$suffix = isset( $item['suffix'] ) ? (string) $item['suffix'] : '';
			$attrs  = wp_json_encode( array(
				'value'    => $value,
				'suffix'   => $suffix,
				'label'    => $label,
				'duration' => 1500,
			) );
			$blocks[] = '<!-- wp:column {"className":"rms-stat"} -->';
			$blocks[] = '<div class="wp-block-column rms-stat">';
			$blocks[] = '<!-- wp:rms/counter ' . $attrs . ' /-->';
			$blocks[] = '</div>';
			$blocks[] = '<!-- /wp:column -->';
		}
		$blocks[] = '</div>';
		$blocks[] = '<!-- /wp:columns -->';
		$blocks[] = '</div>';
		$blocks[] = '<!-- /wp:group -->';
		$blocks[] = '</div>';
		$blocks[] = '<!-- /wp:group -->';

		return implode( "\n", $blocks );
	}

	/**
	 * Setup front page content based on current language.
	 */
	public static function setup_front_page_content() {
		if ( ! is_front_page() || ! function_exists( 'pll_current_language' ) ) {
			return;
		}

		$current_lang = pll_current_language();
		$content      = self::$default_content[ $current_lang ] ?? self::$default_content['en'];

		// Set dynamic content for the current language.
		add_filter(
			'pre_get_document_title',
			function () use ( $content ) {
				return $content['site_title'] . ' - ' . $content['site_description'];
			}
		);
	}

	/**
	 * Filter front page content to ensure proper language display.
	 *
	 * @param string $content Content.
	 * @return string Modified content.
	 */
	public static function filter_front_page_content( $content ) {
		if ( ! is_front_page() || ! function_exists( 'pll_current_language' ) ) {
			return $content;
		}

		// If content is empty or default, generate language-specific content.
		if ( empty( $content ) || false !== strpos( $content, 'Coming Soon' ) ) {
			$current_lang = pll_current_language();
			return self::generate_front_page_content( $current_lang );
		}

		return $content;
	}

	/**
	 * Add language-specific meta tags.
	 */
	public static function add_language_meta_tags() {
		if ( ! function_exists( 'pll_current_language' ) ) {
			return;
		}

		$current_lang = pll_current_language();
		$content      = self::$default_content[ $current_lang ] ?? self::$default_content['en'];

		echo '<meta name="description" content="' . esc_attr( $content['site_description'] ) . '">' . "\n";
		echo '<meta property="og:title" content="' . esc_attr( $content['site_title'] ) . '">' . "\n";
		echo '<meta property="og:description" content="' . esc_attr( $content['site_description'] ) . '">' . "\n";
		echo '<meta name="twitter:title" content="' . esc_attr( $content['site_title'] ) . '">' . "\n";
		echo '<meta name="twitter:description" content="' . esc_attr( $content['site_description'] ) . '">' . "\n";
	}

	/**
	 * Reset multilingual pages (for development/testing).
	 */
	public static function reset_multilingual_pages() {
		delete_option( 'rms_multilingual_pages_created' );
	}
}

// Initialize the multilingual content system.
RMS_Multilingual_Content::init();
