<?php
// phpcs:ignoreFile

/**
 * Multilingual Content Management for RMS Theme
 *
 * This file handles automatic content generation and translation
 * for different languages supported by Polylang.
 *
 * @package RMS
 * @since 1.0.0
 */

// Prevent direct access.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Class for managing multilingual content
 */
class RMS_Multilingual_Content {


	/**
	 * Default content for different languages
	 */
	private static $default_content = array(
		'en' => array(
			'site_title'        => 'Rimess Almasry',
			'site_description'  => 'Excellence in Multiple Fields',
			'hero_title'        => 'Welcome to Excellence',
			'hero_subtitle'     => 'Discover our premium services and innovative solutions',
			'hero_cta'          => 'Get Started',
			// Header CTA banner
			'header_cta_title'    => 'Apply Now',
			'header_cta_subtitle' => "(Professional Master's & PhD)",
			'header_cta_button'   => 'Register here',
			'header_cta_url'      => '/application-form/',
			'services_title'    => 'Our Services',
			'services_subtitle' => 'Professional solutions tailored to your needs',
			'about_title'       => 'About Us',
			'about_content'     => 'We are dedicated to providing exceptional services across multiple industries. Our commitment to excellence drives everything we do.',
			'contact_title'     => 'Get in Touch',
			'contact_subtitle'  => 'Ready to start your journey with us?',
			'footer_text'       => 'Excellence in Multiple Fields',
			'copyright'         => '© 2025 Rimess Almasry. All rights reserved.',
		),
		'fr' => array(
			'site_title'        => 'Rimess Almasry',
			'site_description'  => 'L\'Excellence dans Plusieurs Domaines',
			'hero_title'        => 'Bienvenue à l\'Excellence',
			'hero_subtitle'     => 'Découvrez nos services premium et solutions innovantes',
			'hero_cta'          => 'Commencer',
			// Header CTA banner
			'header_cta_title'    => 'Postulez maintenant',
			'header_cta_subtitle' => '(Master et Doctorat Professionnels)',
			'header_cta_button'   => 'Inscrivez-vous ici',
			'header_cta_url'      => '/fr/application-form/',
			'services_title'    => 'Nos Services',
			'services_subtitle' => 'Solutions professionnelles adaptées à vos besoins',
			'about_title'       => 'À Propos',
			'about_content'     => 'Nous nous consacrons à fournir des services exceptionnels dans plusieurs industries. Notre engagement envers l\'excellence guide tout ce que nous faisons.',
			'contact_title'     => 'Contactez-nous',
			'contact_subtitle'  => 'Prêt à commencer votre parcours avec nous?',
			'footer_text'       => 'L\'Excellence dans Plusieurs Domaines',
			'copyright'         => '© 2025 Rimess Almasry. Tous droits réservés.',
		),
		'ar' => array(
			'site_title'        => 'ريماس المصري',
			'site_description'  => 'التميز في مجالات متعددة',
			'hero_title'        => 'مرحباً بكم في التميز',
			'hero_subtitle'     => 'اكتشف خدماتنا المتميزة وحلولنا المبتكرة',
			'hero_cta'          => 'ابدأ الآن',
			// Header CTA banner
			'header_cta_title'    => 'تقدم الآن',
			'header_cta_subtitle' => '(الماجستير والدكتوراه المهنية )',
			'header_cta_button'   => 'التسجيل هنا',
			'header_cta_url'      => '/ar/application-form/',
			'services_title'    => 'خدماتنا',
			'services_subtitle' => 'حلول مهنية مصممة خصيصاً لاحتياجاتك',
			'about_title'       => 'من نحن',
			'about_content'     => 'نحن ملتزمون بتقديم خدمات استثنائية عبر صناعات متعددة. التزامنا بالتميز يوجه كل ما نقوم به.',
			'contact_title'     => 'تواصل معنا',
			'contact_subtitle'  => 'مستعد لبدء رحلتك معنا؟',
			'footer_text'       => 'التميز في مجالات متعددة',
			'copyright'         => '© 2025 ريماس المصري. جميع الحقوق محفوظة.',
		),
	);

	/**
	 * Service cards content for different languages
	 */
	private static $service_cards = array(
		'en' => array(
			array(
				'icon'        => '🚀',
				'title'       => 'Innovation',
				'description' => 'Cutting-edge solutions powered by the latest technology and creative thinking.',
				'link'        => '/services/innovation/',
			),
			array(
				'icon'        => '💼',
				'title'       => 'Business Solutions',
				'description' => 'Comprehensive business strategies to help your company grow and succeed.',
				'link'        => '/services/business/',
			),
			array(
				'icon'        => '🎯',
				'title'       => 'Consulting',
				'description' => 'Expert advice and guidance to navigate complex challenges and opportunities.',
				'link'        => '/services/consulting/',
			),
			array(
				'icon'        => '🔧',
				'title'       => 'Technical Support',
				'description' => 'Professional technical assistance and maintenance for all your systems.',
				'link'        => '/services/support/',
			),
		),
		'fr' => array(
			array(
				'icon'        => '🚀',
				'title'       => 'Innovation',
				'description' => 'Solutions de pointe alimentées par la dernière technologie et la pensée créative.',
				'link'        => '/fr/services/innovation/',
			),
			array(
				'icon'        => '💼',
				'title'       => 'Solutions Business',
				'description' => 'Stratégies commerciales complètes pour aider votre entreprise à croître et réussir.',
				'link'        => '/fr/services/business/',
			),
			array(
				'icon'        => '🎯',
				'title'       => 'Conseil',
				'description' => 'Conseils d\'experts et orientation pour naviguer dans les défis et opportunités complexes.',
				'link'        => '/fr/services/consulting/',
			),
			array(
				'icon'        => '🔧',
				'title'       => 'Support Technique',
				'description' => 'Assistance technique professionnelle et maintenance pour tous vos systèmes.',
				'link'        => '/fr/services/support/',
			),
		),
		'ar' => array(
			array(
				'icon'        => '🚀',
				'title'       => 'الابتكار',
				'description' => 'حلول متطورة مدعومة بأحدث التقنيات والتفكير الإبداعي.',
				'link'        => '/ar/services/innovation/',
			),
			array(
				'icon'        => '💼',
				'title'       => 'الحلول التجارية',
				'description' => 'استراتيجيات تجارية شاملة لمساعدة شركتك على النمو والنجاح.',
				'link'        => '/ar/services/business/',
			),
			array(
				'icon'        => '🎯',
				'title'       => 'الاستشارات',
				'description' => 'نصائح وإرشادات الخبراء للتنقل في التحديات والفرص المعقدة.',
				'link'        => '/ar/services/consulting/',
			),
			array(
				'icon'        => '🔧',
				'title'       => 'الدعم التقني',
				'description' => 'مساعدة تقنية مهنية وصيانة لجميع أنظمتك.',
				'link'        => '/ar/services/support/',
			),
		),
	);

	/**
	 * Initialize the multilingual content system
	 */
	public static function init() {
		add_action( 'init', array( __CLASS__, 'setup_translated_pages' ) );
		add_action( 'wp', array( __CLASS__, 'setup_front_page_content' ) );
		add_filter( 'the_content', array( __CLASS__, 'filter_front_page_content' ) );
		add_action( 'wp_head', array( __CLASS__, 'add_language_meta_tags' ) );
	}

	/**
	 * Get content for current language
	 */
	public static function get_content( $key, $lang = null ) {
		if ( ! $lang ) {
			$lang = function_exists( 'pll_current_language' ) ? pll_current_language() : 'en';
		}

		// Fallback to English if language not found
		if ( ! isset( self::$default_content[ $lang ] ) ) {
			$lang = 'en';
		}

		return self::$default_content[ $lang ][ $key ] ?? '';
	}

	/**
	 * Get service cards for current language
	 */
	public static function get_service_cards( $lang = null ) {
		if ( ! $lang ) {
			$lang = function_exists( 'pll_current_language' ) ? pll_current_language() : 'en';
		}

		// Fallback to English if language not found
		if ( ! isset( self::$service_cards[ $lang ] ) ) {
			$lang = 'en';
		}

		return self::$service_cards[ $lang ] ?? array();
	}

	/**
	 * Setup translated pages automatically
	 */
	public static function setup_translated_pages() {
		if ( ! function_exists( 'pll_languages_list' ) ) {
			return;
		}

		// Only run this once
		if ( get_option( 'rms_multilingual_pages_created' ) ) {
			return;
		}

		$languages      = pll_languages_list();
		$front_page_ids = array();

		foreach ( $languages as $lang ) {
			$page_id = self::create_translated_front_page( $lang );
			if ( $page_id ) {
				$front_page_ids[ $lang ] = $page_id;
			}
		}

		// Set front pages for each language
		if ( ! empty( $front_page_ids ) ) {
			foreach ( $front_page_ids as $lang => $page_id ) {
				if ( function_exists( 'pll_set_post_language' ) ) {
					pll_set_post_language( $page_id, $lang );
				}
			}

			// Link the translations
			if ( function_exists( 'pll_save_post_translations' ) && count( $front_page_ids ) > 1 ) {
				pll_save_post_translations( $front_page_ids );
			}
		}

		update_option( 'rms_multilingual_pages_created', true );
	}

	/**
	 * Create a translated front page
	 */
	private static function create_translated_front_page( $lang ) {
		$content = self::$default_content[ $lang ] ?? self::$default_content['en'];

		$page_content = self::generate_front_page_content( $lang );

		$page_data = array(
			'post_title'   => $content['site_title'],
			'post_content' => $page_content,
			'post_status'  => 'publish',
			'post_type'    => 'page',
			'post_name'    => $lang === 'en' ? 'home' : "home-{$lang}",
			'meta_input'   => array(
				'_wp_page_template' => 'index.php',
			),
		);

		$page_id = wp_insert_post( $page_data );

		if ( $page_id && ! is_wp_error( $page_id ) ) {
			// Set as front page for this language if it's the default language
			if ( $lang === pll_default_language() ) {
				update_option( 'page_on_front', $page_id );
				update_option( 'show_on_front', 'page' );
			}

			return $page_id;
		}

		return false;
	}

	/**
	 * Generate front page content with blocks
	 */
	private static function generate_front_page_content( $lang ) {
		$content       = self::$default_content[ $lang ] ?? self::$default_content['en'];
		$service_cards = self::get_service_cards( $lang );

		$blocks = array();

		// Hero Section
		$blocks[] = '<!-- wp:group {"align":"full","style":{"spacing":{"padding":{"top":"4rem","bottom":"4rem"}}},"backgroundColor":"black","textColor":"white"} -->';
		$blocks[] = '<div class="wp-block-group alignfull has-white-color has-black-background-color has-text-color has-background" style="padding-top:4rem;padding-bottom:4rem">';
		$blocks[] = '<!-- wp:heading {"textAlign":"center","level":1,"fontSize":"huge"} -->';
		$blocks[] = '<h1 class="wp-block-heading has-text-align-center has-huge-font-size">' . esc_html( $content['hero_title'] ) . '</h1>';
		$blocks[] = '<!-- /wp:heading -->';
		$blocks[] = '<!-- wp:paragraph {"align":"center","fontSize":"large"} -->';
		$blocks[] = '<p class="has-text-align-center has-large-font-size">' . esc_html( $content['hero_subtitle'] ) . '</p>';
		$blocks[] = '<!-- /wp:paragraph -->';
		$blocks[] = '<!-- wp:buttons {"layout":{"type":"flex","justifyContent":"center"}} -->';
		$blocks[] = '<div class="wp-block-buttons">';
		$blocks[] = '<!-- wp:button {"backgroundColor":"gold","textColor":"black"} -->';
		$blocks[] = '<div class="wp-block-button"><a class="wp-block-button__link has-black-color has-gold-background-color has-text-color has-background wp-element-button">' . esc_html( $content['hero_cta'] ) . '</a></div>';
		$blocks[] = '<!-- /wp:button -->';
		$blocks[] = '</div>';
		$blocks[] = '<!-- /wp:buttons -->';
		$blocks[] = '</div>';
		$blocks[] = '<!-- /wp:group -->';

		// Services Section
		$blocks[] = '<!-- wp:group {"align":"full","style":{"spacing":{"padding":{"top":"4rem","bottom":"4rem"}}}} -->';
		$blocks[] = '<div class="wp-block-group alignfull" style="padding-top:4rem;padding-bottom:4rem">';
		$blocks[] = '<!-- wp:heading {"textAlign":"center","level":2} -->';
		$blocks[] = '<h2 class="wp-block-heading has-text-align-center">' . esc_html( $content['services_title'] ) . '</h2>';
		$blocks[] = '<!-- /wp:heading -->';
		$blocks[] = '<!-- wp:paragraph {"align":"center"} -->';
		$blocks[] = '<p class="has-text-align-center">' . esc_html( $content['services_subtitle'] ) . '</p>';
		$blocks[] = '<!-- /wp:paragraph -->';

		// Service Cards
		$blocks[] = '<!-- wp:columns {"align":"wide"} -->';
		$blocks[] = '<div class="wp-block-columns alignwide">';

		foreach ( $service_cards as $card ) {
			$blocks[] = '<!-- wp:column -->';
			$blocks[] = '<div class="wp-block-column">';
			$blocks[] = '<!-- wp:rms/service-card {"icon":"' . esc_attr( $card['icon'] ) . '","title":"' . esc_attr( $card['title'] ) . '","description":"' . esc_attr( $card['description'] ) . '","linkUrl":"' . esc_url( $card['link'] ) . '"} /-->';
			$blocks[] = '</div>';
			$blocks[] = '<!-- /wp:column -->';
		}

		$blocks[] = '</div>';
		$blocks[] = '<!-- /wp:columns -->';
		$blocks[] = '</div>';
		$blocks[] = '<!-- /wp:group -->';

		return implode( "\n", $blocks );
	}

	/**
	 * Setup front page content based on current language
	 */
	public static function setup_front_page_content() {
		if ( ! is_front_page() || ! function_exists( 'pll_current_language' ) ) {
			return;
		}

		$current_lang = pll_current_language();
		$content      = self::$default_content[ $current_lang ] ?? self::$default_content['en'];

		// Set dynamic content for the current language
		add_filter(
			'pre_get_document_title',
			function () use ( $content ) {
				return $content['site_title'] . ' - ' . $content['site_description'];
			}
		);
	}

	/**
	 * Filter front page content to ensure proper language display
	 */
	public static function filter_front_page_content( $content ) {
		if ( ! is_front_page() || ! function_exists( 'pll_current_language' ) ) {
			return $content;
		}

		// If content is empty or default, generate language-specific content
		if ( empty( $content ) || strpos( $content, 'Coming Soon' ) !== false ) {
			$current_lang = pll_current_language();
			return self::generate_front_page_content( $current_lang );
		}

		return $content;
	}

	/**
	 * Add language-specific meta tags
	 */
	public static function add_language_meta_tags() {
		if ( ! function_exists( 'pll_current_language' ) ) {
			return;
		}

		$current_lang = pll_current_language();
		$content      = self::$default_content[ $current_lang ] ?? self::$default_content['en'];

		echo '<meta name="description" content="' . esc_attr( $content['site_description'] ) . '">' . "\n";
		echo '<meta property="og:title" content="' . esc_attr( $content['site_title'] ) . '">' . "\n";
		echo '<meta property="og:description" content="' . esc_attr( $content['site_description'] ) . '">' . "\n";
		echo '<meta name="twitter:title" content="' . esc_attr( $content['site_title'] ) . '">' . "\n";
		echo '<meta name="twitter:description" content="' . esc_attr( $content['site_description'] ) . '">' . "\n";
	}

	/**
	 * Reset multilingual pages (for development/testing)
	 */
	public static function reset_multilingual_pages() {
		delete_option( 'rms_multilingual_pages_created' );
	}
}

// Initialize the multilingual content system
RMS_Multilingual_Content::init();

/**
 * Helper functions for templates
 */

/**
 * Get translated content
 */
function rms_get_content( $key, $lang = null ) {
	return RMS_Multilingual_Content::get_content( $key, $lang );
}

/**
 * Get service cards for current language
 */
function rms_get_service_cards( $lang = null ) {
	return RMS_Multilingual_Content::get_service_cards( $lang );
}

/**
 * Display translated content
 */
function rms_content( $key, $lang = null ) {
	echo esc_html( rms_get_content( $key, $lang ) );
}
