<?php
// phpcs:ignoreFile

/**
 * RMS Theme Security Functions
 * Security & Performance Agent
 *
 * @package RMS
 */

// Prevent direct access.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Security enhancements for RMS theme
 */
class RMS_Security {


	/**
	 * Initialize security features
	 */
	public static function init() {
		add_action( 'init', array( __CLASS__, 'setup_security_headers' ) );
		add_filter( 'wp_headers', array( __CLASS__, 'add_security_headers' ) );
		add_action( 'wp_enqueue_scripts', array( __CLASS__, 'secure_enqueue_scripts' ) );
		add_filter( 'script_loader_tag', array( __CLASS__, 'add_script_security_attributes' ), 10, 2 );
	}

	/**
	 * Setup security headers
	 */
	public static function setup_security_headers() {
		// Remove WordPress version from head
		remove_action( 'wp_head', 'wp_generator' );

		// Remove version from RSS feeds
		add_filter( 'the_generator', '__return_empty_string' );

		// Disable XML-RPC if not needed
		add_filter( 'xmlrpc_enabled', '__return_false' );

		// Remove unnecessary REST API links from head
		remove_action( 'wp_head', 'rest_output_link_wp_head' );
		remove_action( 'wp_head', 'wp_oembed_add_discovery_links' );
	}

	/**
	 * Add security headers
	 */
	public static function add_security_headers( $headers ) {
		// X-Content-Type-Options
		$headers['X-Content-Type-Options'] = 'nosniff';

		// X-Frame-Options
		$headers['X-Frame-Options'] = 'SAMEORIGIN';

		// X-XSS-Protection
		$headers['X-XSS-Protection'] = '1; mode=block';

		// Referrer Policy
		$headers['Referrer-Policy'] = 'strict-origin-when-cross-origin';

		return $headers;
	}

	/**
	 * Secure script enqueuing with integrity checks
	 */
	public static function secure_enqueue_scripts() {
		// Only load scripts where needed
		if ( ! is_admin() ) {
			// Enqueue main theme script with proper dependencies
			wp_enqueue_script(
				'rms-main',
				get_template_directory_uri() . '/assets/js/main.js',
				array( 'jquery' ),
				wp_get_theme()->get( 'Version' ),
				true
			);

			// Add nonce for AJAX requests
			wp_localize_script(
				'rms-main',
				'rms_ajax',
				array(
					'ajax_url' => admin_url( 'admin-ajax.php' ),
					'nonce'    => wp_create_nonce( 'rms_ajax_nonce' ),
				)
			);
		}
	}

	/**
	 * Add security attributes to script tags
	 */
	public static function add_script_security_attributes( $tag, $handle ) {
		// Add integrity and crossorigin attributes for external scripts
		if ( strpos( $tag, 'external-script' ) !== false ) {
			$tag = str_replace( ' src=', ' integrity="sha384-..." crossorigin="anonymous" src=', $tag );
		}

		return $tag;
	}

	/**
	 * Sanitize user input
	 */
	public static function sanitize_input( $input, $type = 'text' ) {
		switch ( $type ) {
			case 'email':
				return sanitize_email( $input );
			case 'url':
				return esc_url_raw( $input );
			case 'html':
				return wp_kses_post( $input );
			case 'text':
			default:
				return sanitize_text_field( $input );
		}
	}

	/**
	 * Validate nonce
	 */
	public static function verify_nonce( $nonce, $action ) {
		return wp_verify_nonce( $nonce, $action );
	}
}

// Initialize security features
RMS_Security::init();
