<?php

/**
 * Plugin Name: RMS Certificate Verification
 * Description: Verify certificates by number. Provides admin UI to manage certificates, a REST API endpoint, and a frontend shortcode.
 * Version: 1.2.0
 * Author: AH Dev
 * License: GPLv2 or later
 * Text Domain: rms-cv
 * Update URI: https://wordpress-updater.rms-updates.workers.dev
 */

if (! defined('ABSPATH')) {
	exit;
}

class RMS_Certificate_Verification
{
	const VERSION = '1.2.0';
	const TABLE   = 'rms_certificates';
	private static $instance = null;

	public static function instance()
	{
		if (null === self::$instance) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	private function __construct()
	{
		register_activation_hook(__FILE__, [__CLASS__, 'activate']);
		add_action('init', [$this, 'init']);
		add_action('admin_menu', [$this, 'admin_menu']);
		add_action('rest_api_init', [$this, 'register_rest_routes']);
		add_shortcode('rms_verify_certificate', [$this, 'shortcode']);
	}

	public static function activate()
	{
		global $wpdb;
		$table_name   = $wpdb->prefix . self::TABLE;
		$charset_collate = $wpdb->get_charset_collate();

		$sql = "CREATE TABLE {$table_name} (
			id BIGINT(20) UNSIGNED NOT NULL AUTO_INCREMENT,
			cert_number VARCHAR(64) NOT NULL,
			name VARCHAR(191) NULL,
			program VARCHAR(191) NULL,
			issued_on DATE NULL,
			meta LONGTEXT NULL,
			first_name VARCHAR(191) NULL,
			last_name VARCHAR(191) NULL,
			email VARCHAR(191) NULL,
			field_of_study VARCHAR(191) NULL,
			course_category VARCHAR(191) NULL,
			iso_standards VARCHAR(191) NULL,
			exam_date DATE NULL,
			exam_result VARCHAR(32) NULL,
			notes LONGTEXT NULL,
			created_at DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
			updated_at DATETIME NULL ON UPDATE CURRENT_TIMESTAMP,
			UNIQUE KEY cert_number (cert_number),
			PRIMARY KEY  (id)
		) {$charset_collate};";

		require_once ABSPATH . 'wp-admin/includes/upgrade.php';
		dbDelta($sql);
		update_option('rms_cv_db_version', '1.1.0');
	}

	public function init()
	{
		// i18n
		load_plugin_textdomain('rms-cv', false, dirname(plugin_basename(__FILE__)) . '/languages');
		// Fallback: if MO files are not in /languages or global WP directory, try plugin root.
		$locale = function_exists('pll_current_language') ? pll_current_language('locale') : get_locale();
		$mo_root = __DIR__ . '/rms-cv-' . $locale . '.mo';
		if (file_exists($mo_root)) {
			load_textdomain('rms-cv', $mo_root);
		}
		$this->maybe_upgrade_schema();

		// Initialize automatic updates via Cloudflare Worker if configured.
		if ( defined( 'RMS_UPDATER_URL' ) && RMS_UPDATER_URL ) {
			require_once __DIR__ . '/includes/class-rms-plugin-updater.php';
			$plugin_file = plugin_basename( __FILE__ );
			new RMS_Plugin_Updater( $plugin_file, 'rms-certificate-verification', self::VERSION, RMS_UPDATER_URL );
		}
	}

	private function maybe_upgrade_schema()
	{
		$version = get_option('rms_cv_db_version', '1.0.0');
		if (version_compare($version, '1.1.0', '<')) {
			global $wpdb;
			$table_name   = $wpdb->prefix . self::TABLE;
			$charset_collate = $wpdb->get_charset_collate();
			$sql = "CREATE TABLE {$table_name} (
				id BIGINT(20) UNSIGNED NOT NULL AUTO_INCREMENT,
				cert_number VARCHAR(64) NOT NULL,
				name VARCHAR(191) NULL,
				program VARCHAR(191) NULL,
				issued_on DATE NULL,
				meta LONGTEXT NULL,
				first_name VARCHAR(191) NULL,
				last_name VARCHAR(191) NULL,
				email VARCHAR(191) NULL,
				field_of_study VARCHAR(191) NULL,
				course_category VARCHAR(191) NULL,
				iso_standards VARCHAR(191) NULL,
				exam_date DATE NULL,
				exam_result VARCHAR(32) NULL,
				notes LONGTEXT NULL,
				created_at DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
				updated_at DATETIME NULL ON UPDATE CURRENT_TIMESTAMP,
				UNIQUE KEY cert_number (cert_number),
				PRIMARY KEY  (id)
			) {$charset_collate};";
			require_once ABSPATH . 'wp-admin/includes/upgrade.php';
			dbDelta($sql);
			update_option('rms_cv_db_version', '1.1.0');
		}
	}

	public function admin_menu()
	{
		add_menu_page(
			__('Certificate Verification', 'rms-cv'),
			__('Certificates', 'rms-cv'),
			'manage_options',
			'rms-cv',
			[$this, 'render_admin_page'],
			'dashicons-yes',
			56
		);
	}

	private function table()
	{
		global $wpdb;
		return $wpdb->prefix . self::TABLE;
	}

	public function render_admin_page()
	{
		if (! current_user_can('manage_options')) {
			wp_die(esc_html__('Insufficient permissions.', 'rms-cv'));
		}

		$action = isset($_POST['rms_cv_action']) ? sanitize_text_field(wp_unslash($_POST['rms_cv_action'])) : '';
		if ('create' === $action && check_admin_referer('rms_cv_create', 'rms_cv_nonce')) {
			$this->handle_create();
		}
		if ('update' === $action && check_admin_referer('rms_cv_update', 'rms_cv_nonce')) {
			$this->handle_update();
		}

		$edit_id = isset($_GET['edit']) ? intval($_GET['edit']) : 0; // phpcs:ignore WordPress.Security.NonceVerification.Recommended
		if ($edit_id > 0) {
			$this->render_edit_screen($edit_id);
			return;
		}

		$this->render_admin_screen();
	}

	private function handle_create()
	{
		global $wpdb;
		$cert_number = isset($_POST['cert_number']) ? sanitize_text_field(wp_unslash($_POST['cert_number'])) : '';
		$issued_on   = isset($_POST['issued_on']) ? sanitize_text_field(wp_unslash($_POST['issued_on'])) : '';
		$notes       = isset($_POST['notes']) ? wp_kses_post(wp_unslash($_POST['notes'])) : '';
		$first_name      = isset($_POST['first_name']) ? sanitize_text_field(wp_unslash($_POST['first_name'])) : '';
		$last_name       = isset($_POST['last_name']) ? sanitize_text_field(wp_unslash($_POST['last_name'])) : '';
		$email           = isset($_POST['email']) ? sanitize_email(wp_unslash($_POST['email'])) : '';
		$field_of_study  = isset($_POST['field_of_study']) ? sanitize_text_field(wp_unslash($_POST['field_of_study'])) : '';
		$course_category = isset($_POST['course_category']) ? sanitize_text_field(wp_unslash($_POST['course_category'])) : '';
		$iso_standards   = isset($_POST['iso_standards']) ? sanitize_text_field(wp_unslash($_POST['iso_standards'])) : '';
		$exam_date       = isset($_POST['exam_date']) ? sanitize_text_field(wp_unslash($_POST['exam_date'])) : '';
		$exam_result     = isset($_POST['exam_result']) ? sanitize_text_field(wp_unslash($_POST['exam_result'])) : '';

		if (empty($cert_number) || empty($first_name) || empty($field_of_study) || empty($exam_date) || empty($issued_on) || empty($exam_result)) {
			add_settings_error('rms-cv', 'missing_required', __('Please fill all required fields.', 'rms-cv'), 'error');
			return;
		}
		$exam_date = $this->normalize_date($exam_date);
		$issued_on = $this->normalize_date($issued_on);

		$inserted = $wpdb->insert(
			$this->table(),
			[
				'cert_number' => $cert_number,
				'issued_on'   => $issued_on ?: null,
				'meta'        => $notes,
				'first_name'      => $first_name,
				'last_name'       => $last_name,
				'email'           => $email,
				'field_of_study'  => $field_of_study,
				'course_category' => $course_category,
				'iso_standards'   => $iso_standards,
				'exam_date'       => $exam_date ?: null,
				'exam_result'     => $exam_result,
				'notes'           => $notes,
			],
			['%s', '%s', '%s', '%s', '%s', '%s', '%s', '%s', '%s', '%s', '%s', '%s']
		);

		if (false === $inserted) {
			add_settings_error('rms-cv', 'db_error', __('Could not save certificate. It may already exist.', 'rms-cv'), 'error');
		} else {
			add_settings_error('rms-cv', 'saved', __('Certificate saved.', 'rms-cv'), 'updated');
		}
	}

	private function render_admin_screen()
	{
		global $wpdb;
		$certs = $wpdb->get_results('SELECT id, cert_number, first_name, last_name, field_of_study, exam_result, issued_on, created_at FROM ' . $this->table() . ' ORDER BY id DESC LIMIT 100');
		settings_errors('rms-cv');
?>
		<div class="wrap">
			<h1><?php echo esc_html(get_admin_page_title()); ?></h1>
			<h2><?php esc_html_e('Add Certificate', 'rms-cv'); ?></h2>
			<form method="post">
				<?php wp_nonce_field('rms_cv_create', 'rms_cv_nonce'); ?>
				<input type="hidden" name="rms_cv_action" value="create" />
				<table class="form-table" role="presentation">
					<tr>
						<th scope="row"><label for="cert_number"><?php esc_html_e('Certificate Number', 'rms-cv'); ?></label></th>
						<td><input type="text" id="cert_number" name="cert_number" class="regular-text" required /></td>
					</tr>
					<tr>
						<th scope="row"><label for="first_name"><?php esc_html_e('First Name', 'rms-cv'); ?></label></th>
						<td><input type="text" id="first_name" name="first_name" class="regular-text" required /></td>
					</tr>
					<tr>
						<th scope="row"><label for="last_name"><?php esc_html_e('Last Name', 'rms-cv'); ?></label></th>
						<td><input type="text" id="last_name" name="last_name" class="regular-text" /></td>
					</tr>
					<tr>
						<th scope="row"><label for="email"><?php esc_html_e('Email', 'rms-cv'); ?></label></th>
						<td><input type="email" id="email" name="email" class="regular-text" /></td>
					</tr>
					<tr>
						<th scope="row"><label for="field_of_study"><?php esc_html_e('Field of Study', 'rms-cv'); ?></label></th>
						<td><input type="text" id="field_of_study" name="field_of_study" class="regular-text" required /></td>
					</tr>
					<tr>
						<th scope="row"><label for="course_category"><?php esc_html_e('Course Category', 'rms-cv'); ?></label></th>
						<td><input type="text" id="course_category" name="course_category" class="regular-text" /></td>
					</tr>
					<tr>
						<th scope="row"><label for="iso_standards"><?php esc_html_e('ISO Standards', 'rms-cv'); ?></label></th>
						<td><input type="text" id="iso_standards" name="iso_standards" class="regular-text" /></td>
					</tr>
					<tr>
						<th scope="row"><label for="exam_date"><?php esc_html_e('Exam Date', 'rms-cv'); ?></label></th>
						<td><input type="date" id="exam_date" name="exam_date" required /></td>
					</tr>
					<tr>
						<th scope="row"><label for="issued_on"><?php esc_html_e('Issue Date', 'rms-cv'); ?></label></th>
						<td><input type="date" id="issued_on" name="issued_on" required /></td>
					</tr>
					<tr>
						<th scope="row"><label for="exam_result"><?php esc_html_e('Exam Result', 'rms-cv'); ?></label></th>
						<td>
							<select id="exam_result" name="exam_result" class="regular-text" required>
								<option value=""><?php esc_html_e('Select Result', 'rms-cv'); ?></option>
								<option value="pass"><?php esc_html_e('Pass', 'rms-cv'); ?></option>
								<option value="fail"><?php esc_html_e('Fail', 'rms-cv'); ?></option>
								<option value="pending"><?php esc_html_e('Pending', 'rms-cv'); ?></option>
							</select>
						</td>
					</tr>
					<tr>
						<th scope="row"><label for="notes"><?php esc_html_e('Notes (optional)', 'rms-cv'); ?></label></th>
						<td><textarea id="notes" name="notes" class="large-text" rows="3"></textarea></td>
					</tr>
				</table>
				<?php submit_button(__('Save Certificate', 'rms-cv')); ?>
			</form>

			<h2 style="margin-top:2rem;"><?php esc_html_e('Recent Certificates', 'rms-cv'); ?></h2>
			<table class="widefat striped">
				<thead>
					<tr>
						<th><?php esc_html_e('Number', 'rms-cv'); ?></th>
						<th><?php esc_html_e('Name', 'rms-cv'); ?></th>
						<th><?php esc_html_e('Field of Study', 'rms-cv'); ?></th>
						<th><?php esc_html_e('Exam Result', 'rms-cv'); ?></th>
						<th><?php esc_html_e('Issue Date', 'rms-cv'); ?></th>
						<th><?php esc_html_e('Created', 'rms-cv'); ?></th>
						<th><?php esc_html_e('Actions', 'rms-cv'); ?></th>
					</tr>
				</thead>
				<tbody>
					<?php if ($certs) : foreach ($certs as $c) : ?>
							<tr>
								<td><?php echo esc_html($c->cert_number); ?></td>
								<td><?php echo esc_html(trim($c->first_name . ' ' . $c->last_name)); ?></td>
								<td><?php echo esc_html($c->field_of_study); ?></td>
								<td><?php echo esc_html($c->exam_result); ?></td>
								<td><?php echo esc_html($c->issued_on); ?></td>
								<td><?php echo esc_html($c->created_at); ?></td>
								<td><a href="<?php echo esc_url(admin_url('admin.php?page=rms-cv&edit=' . intval($c->id))); ?>"><?php esc_html_e('Edit', 'rms-cv'); ?></a></td>
							</tr>
						<?php endforeach;
					else : ?>
						<tr>
							<td colspan="7"><?php esc_html_e('No certificates yet.', 'rms-cv'); ?></td>
						</tr>
					<?php endif; ?>
				</tbody>
			</table>
		</div>
	<?php
	}

	private function render_edit_screen($id)
	{
		global $wpdb;
		$row = $wpdb->get_row($wpdb->prepare('SELECT * FROM ' . $this->table() . ' WHERE id = %d', $id));
		settings_errors('rms-cv');
	?>
		<div class="wrap">
			<h1><?php esc_html_e('Edit Certificate', 'rms-cv'); ?></h1>
			<p><a href="<?php echo esc_url(admin_url('admin.php?page=rms-cv')); ?>">&larr; <?php esc_html_e('Back to list', 'rms-cv'); ?></a></p>
			<?php if (! $row) : ?>
				<div class="notice notice-error">
					<p><?php esc_html_e('Certificate not found.', 'rms-cv'); ?></p>
				</div>
			<?php else : ?>
				<form method="post">
					<?php wp_nonce_field('rms_cv_update', 'rms_cv_nonce'); ?>
					<input type="hidden" name="rms_cv_action" value="update" />
					<input type="hidden" name="id" value="<?php echo esc_attr($row->id); ?>" />
					<table class="form-table" role="presentation">
						<tr>
							<th scope="row"><label for="cert_number"><?php esc_html_e('Certificate Number', 'rms-cv'); ?></label></th>
							<td><input type="text" id="cert_number" name="cert_number" class="regular-text" required value="<?php echo esc_attr($row->cert_number); ?>" /></td>
						</tr>
						<tr>
							<th scope="row"><label for="first_name"><?php esc_html_e('First Name', 'rms-cv'); ?></label></th>
							<td><input type="text" id="first_name" name="first_name" class="regular-text" required value="<?php echo esc_attr($row->first_name); ?>" /></td>
						</tr>
						<tr>
							<th scope="row"><label for="last_name"><?php esc_html_e('Last Name', 'rms-cv'); ?></label></th>
							<td><input type="text" id="last_name" name="last_name" class="regular-text" value="<?php echo esc_attr($row->last_name); ?>" /></td>
						</tr>
						<tr>
							<th scope="row"><label for="email"><?php esc_html_e('Email', 'rms-cv'); ?></label></th>
							<td><input type="email" id="email" name="email" class="regular-text" value="<?php echo esc_attr($row->email); ?>" /></td>
						</tr>
						<tr>
							<th scope="row"><label for="field_of_study"><?php esc_html_e('Field of Study', 'rms-cv'); ?></label></th>
							<td><input type="text" id="field_of_study" name="field_of_study" class="regular-text" required value="<?php echo esc_attr($row->field_of_study); ?>" /></td>
						</tr>
						<tr>
							<th scope="row"><label for="course_category"><?php esc_html_e('Course Category', 'rms-cv'); ?></label></th>
							<td><input type="text" id="course_category" name="course_category" class="regular-text" value="<?php echo esc_attr($row->course_category); ?>" /></td>
						</tr>
						<tr>
							<th scope="row"><label for="iso_standards"><?php esc_html_e('ISO Standards', 'rms-cv'); ?></label></th>
							<td><input type="text" id="iso_standards" name="iso_standards" class="regular-text" value="<?php echo esc_attr($row->iso_standards); ?>" /></td>
						</tr>
						<tr>
							<th scope="row"><label for="exam_date"><?php esc_html_e('Exam Date', 'rms-cv'); ?></label></th>
							<td><input type="date" id="exam_date" name="exam_date" required value="<?php echo esc_attr($row->exam_date); ?>" /></td>
						</tr>
						<tr>
							<th scope="row"><label for="issued_on"><?php esc_html_e('Issue Date', 'rms-cv'); ?></label></th>
							<td><input type="date" id="issued_on" name="issued_on" required value="<?php echo esc_attr($row->issued_on); ?>" /></td>
						</tr>
						<tr>
							<th scope="row"><label for="exam_result"><?php esc_html_e('Exam Result', 'rms-cv'); ?></label></th>
							<td>
								<select id="exam_result" name="exam_result" class="regular-text" required>
									<option value=""><?php esc_html_e('Select Result', 'rms-cv'); ?></option>
									<option value="pass" <?php selected($row->exam_result, 'pass'); ?>><?php esc_html_e('Pass', 'rms-cv'); ?></option>
									<option value="fail" <?php selected($row->exam_result, 'fail'); ?>><?php esc_html_e('Fail', 'rms-cv'); ?></option>
									<option value="pending" <?php selected($row->exam_result, 'pending'); ?>><?php esc_html_e('Pending', 'rms-cv'); ?></option>
								</select>
							</td>
						</tr>
						<tr>
							<th scope="row"><label for="notes"><?php esc_html_e('Notes (optional)', 'rms-cv'); ?></label></th>
							<td><textarea id="notes" name="notes" class="large-text" rows="3"><?php echo esc_textarea($row->notes ? $row->notes : $row->meta); ?></textarea></td>
						</tr>
					</table>
					<?php submit_button(__('Update Certificate', 'rms-cv')); ?>
				</form>
			<?php endif; ?>
		</div>
	<?php
	}

	private function handle_update()
	{
		global $wpdb;
		$id = isset($_POST['id']) ? intval($_POST['id']) : 0;
		if ($id <= 0) {
			add_settings_error('rms-cv', 'invalid_id', __('Invalid certificate.', 'rms-cv'), 'error');
			return;
		}
		$cert_number = isset($_POST['cert_number']) ? sanitize_text_field(wp_unslash($_POST['cert_number'])) : '';
		$issued_on   = isset($_POST['issued_on']) ? sanitize_text_field(wp_unslash($_POST['issued_on'])) : '';
		$notes       = isset($_POST['notes']) ? wp_kses_post(wp_unslash($_POST['notes'])) : '';
		$first_name      = isset($_POST['first_name']) ? sanitize_text_field(wp_unslash($_POST['first_name'])) : '';
		$last_name       = isset($_POST['last_name']) ? sanitize_text_field(wp_unslash($_POST['last_name'])) : '';
		$email           = isset($_POST['email']) ? sanitize_email(wp_unslash($_POST['email'])) : '';
		$field_of_study  = isset($_POST['field_of_study']) ? sanitize_text_field(wp_unslash($_POST['field_of_study'])) : '';
		$course_category = isset($_POST['course_category']) ? sanitize_text_field(wp_unslash($_POST['course_category'])) : '';
		$iso_standards   = isset($_POST['iso_standards']) ? sanitize_text_field(wp_unslash($_POST['iso_standards'])) : '';
		$exam_date       = isset($_POST['exam_date']) ? sanitize_text_field(wp_unslash($_POST['exam_date'])) : '';
		$exam_result     = isset($_POST['exam_result']) ? sanitize_text_field(wp_unslash($_POST['exam_result'])) : '';

		if (empty($cert_number) || empty($first_name) || empty($field_of_study) || empty($exam_date) || empty($issued_on) || empty($exam_result)) {
			add_settings_error('rms-cv', 'missing_required', __('Please fill all required fields.', 'rms-cv'), 'error');
			return;
		}
		$exam_date = $this->normalize_date($exam_date);
		$issued_on = $this->normalize_date($issued_on);

		$data = [
			'cert_number'    => $cert_number,
			'issued_on'      => $issued_on ?: null,
			'meta'           => $notes,
			'first_name'     => $first_name,
			'last_name'      => $last_name,
			'email'          => $email,
			'field_of_study' => $field_of_study,
			'course_category' => $course_category,
			'iso_standards'  => $iso_standards,
			'exam_date'      => $exam_date ?: null,
			'exam_result'    => $exam_result,
			'notes'          => $notes,
		];

		$updated = $wpdb->update(
			$this->table(),
			$data,
			['id' => $id],
			['%s', '%s', '%s', '%s', '%s', '%s', '%s', '%s', '%s', '%s', '%s', '%s'],
			['%d']
		);

		if (false === $updated) {
			add_settings_error('rms-cv', 'db_error', __('Could not update certificate. The number may already exist.', 'rms-cv'), 'error');
		} else {
			add_settings_error('rms-cv', 'updated', __('Certificate updated.', 'rms-cv'), 'updated');
		}
	}

	private function normalize_date($input)
	{
		$input = trim((string) $input);
		if ('' === $input) {
			return null;
		}
		$input = str_replace('.', '-', $input);
		if (preg_match('/^\d{4}-\d{1,2}-\d{1,2}$/', $input)) {
			try {
				$dt = new DateTime($input);
				return $dt->format('Y-m-d');
			} catch (Exception $e) {
				return null;
			}
		}
		if (preg_match('/^\d{1,2}\/\d{1,2}\/\d{4}$/', $input)) {
			list($d, $m, $y) = array_map('intval', explode('/', $input));
			if (checkdate($m, $d, $y)) {
				return sprintf('%04d-%02d-%02d', $y, $m, $d);
			}
		}
		if (preg_match('/^\d{1,2}-\d{1,2}-\d{4}$/', $input)) {
			list($d, $m, $y) = array_map('intval', explode('-', $input));
			if (checkdate($m, $d, $y)) {
				return sprintf('%04d-%02d-%02d', $y, $m, $d);
			}
		}
		return null;
	}

	public function register_rest_routes()
	{
		register_rest_route('rms-cv/v1', '/verify', [
			'methods'  => 'GET',
			'callback' => [$this, 'rest_verify'],
			'permission_callback' => '__return_true',
			'args' => [
				'number' => ['required' => true, 'type' => 'string'],
			],
		]);
	}

	public function rest_verify(WP_REST_Request $req)
	{
		global $wpdb;
		$number = sanitize_text_field($req->get_param('number'));
		if ('' === $number) {
			return new WP_REST_Response(['ok' => false, 'error' => __('Missing number', 'rms-cv')], 400);
		}
		$table = $this->table();
		$row = $wpdb->get_row($wpdb->prepare("SELECT cert_number, first_name, last_name, field_of_study, course_category, iso_standards, exam_date, issued_on, exam_result, notes FROM {$table} WHERE cert_number = %s", $number), ARRAY_A);
		if ($row) {
			return ['ok' => true, 'data' => $row];
		}
		return new WP_REST_Response(['ok' => false, 'error' => __('Not found', 'rms-cv')], 404);
	}

	public function shortcode($atts)
	{
		$atts = shortcode_atts(['placeholder' => __('Enter certificate number', 'rms-cv')], $atts, 'rms_verify_certificate');
		$nonce = wp_create_nonce('wp_rest');
		ob_start();
	?>
		<form class="rms-cv-form" data-endpoint="<?php echo esc_url(rest_url('rms-cv/v1/verify')); ?>" novalidate>
			<div class="form-field">
				<label for="rms-cv-number" class="form-label"><?php esc_html_e('Certificate number', 'rms-cv'); ?></label>
				<input id="rms-cv-number" name="number" type="text" inputmode="numeric" required placeholder="<?php echo esc_attr($atts['placeholder']); ?>" />
			</div>
			<button type="submit"><span class="btn-text"><?php esc_html_e('Verify', 'rms-cv'); ?></span></button>
			<div class="rms-cv-status" aria-live="polite" role="status"></div>
		</form>
		<script>
			window.RMS_CV_NONCE = <?php echo wp_json_encode($nonce); ?>;
		</script>
<?php
		wp_enqueue_script('rms-cv-public', plugins_url('assets/public.js', __FILE__), [], self::VERSION, true);
		wp_enqueue_style('rms-cv-public', plugins_url('assets/public.css', __FILE__), [], self::VERSION);

		// Localize all user-facing JS strings for translation
		wp_localize_script('rms-cv-public', 'RMS_CV_I18N', [
			'checking' => __('Checking…', 'rms-cv'),
			'verify' => __('Verify', 'rms-cv'),
			'please_enter_number' => __('Please enter a certificate number.', 'rms-cv'),
			'certificate_verified' => __('Certificate verified', 'rms-cv'),
			'number' => __('Number', 'rms-cv'),
			'name' => __('Name', 'rms-cv'),
			'field_of_study' => __('Field of Study', 'rms-cv'),
			'course_category' => __('Course Category', 'rms-cv'),
			'iso_standards' => __('ISO Standards', 'rms-cv'),
			'exam_date' => __('Exam Date', 'rms-cv'),
			'issue_date' => __('Issue Date', 'rms-cv'),
			'result' => __('Result', 'rms-cv'),
			'notes' => __('Notes', 'rms-cv'),
			'verification_result' => __('Verification result', 'rms-cv'),
			'pass' => __('Pass', 'rms-cv'),
			'fail' => __('Fail', 'rms-cv'),
			'pending' => __('Pending', 'rms-cv'),
			'verified' => __('Verified', 'rms-cv'),
			'not_found' => __('Not found', 'rms-cv'),
			'no_certificate' => __('No certificate with this number. Please check the number and try again.', 'rms-cv'),
			'network_error' => __('Network error. Please try again.', 'rms-cv'),
		]);
		return ob_get_clean();
	}
}

RMS_Certificate_Verification::instance();
